<?php

namespace Drupal\db_health\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for managing and deleting DB Health log entries.
 */
class DbHealthLogsForm extends FormBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The private temp store for storing selected timestamps.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStore
   */
  protected $tempStore;

  /**
   * Whether the form is in the confirmation step.
   *
   * @var bool
   */
  protected $confirming = FALSE;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * Constructs a new \Drupal\db_health\Form\DbHealthLogsForm object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $tempstore_factory
   *   The private temp store factory.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   */
  public function __construct(Connection $database, PrivateTempStoreFactory $tempstore_factory, DateFormatterInterface $date_formatter) {
    $this->database = $database;
    $this->tempStore = $tempstore_factory->get('db_health');
    $this->dateFormatter = $date_formatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('tempstore.private'),
      $container->get('date.formatter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'db_health_manage_logs_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $selected = $this->tempStore->get('selected_timestamps') ?? [];

    foreach ($selected as $key => $timestamp) {
      $formatted = $this->dateFormatter->format($timestamp, 'short');
      $selected[$key] = $this->t('@ts', ['@ts' => $formatted, '@raw' => $timestamp]);
    }

    // If confirmation step.
    if ($form_state->get('confirm')) {
      $this->confirming = TRUE;

      $form['fieldset'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Confirm Deletion'),
      ];

      $form['fieldset']['message'] = [
        '#markup' => $this->t('Are you sure you want to delete all entries with the following timestamps?'),
      ];

      $form['fieldset']['selected_timestamps'] = [
        '#markup' => '<ul><li>' . implode('</li><li>', $selected) . '</li></ul>',
      ];

      $form['fieldset']['confirm'] = [
        '#type' => 'submit',
        '#value' => $this->t('Confirm Delete'),
      ];

      $form['fieldset']['cancel'] = [
        '#type' => 'submit',
        '#value' => $this->t('Cancel'),
        '#submit' => ['::cancel'],
      ];

      return $form;
    }

    // Load distinct timestamps.
    $query = $this->database->select('db_health_sizes', 'p');
    $query->distinct();
    $query->fields('p', ['timestamp']);
    $result = $query->execute()->fetchCol();

    if (empty($result)) {
      $form['empty'] = ['#markup' => $this->t('No logs found.')];
      return $form;
    }

    $options = array_combine($result, $result);

    foreach ($options as $key => $timestamp) {
      $formatted = $this->dateFormatter->format($timestamp, 'short');
      $options[$key] = $this->t('@ts', ['@ts' => $formatted, '@raw' => $timestamp]);
    }

    $form['fieldset'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Manage DB Health Logs'),
      '#description' => $this->t('Select timestamps to delete associated log entries.'),
    ];
    
    $form['fieldset']['timestamps'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Select timestamps to delete'),
      '#options' => $options,
    ];

    $form['fieldset']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Delete'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    if ($this->confirming) {
      $selected = $this->tempStore->get('selected_timestamps') ?? [];

      if (!empty($selected)) {
        $this->database->delete('db_health_sizes')
          ->condition('timestamp', $selected, 'IN')
          ->execute();

        $this->messenger()->addStatus($this->t('Deleted logs for selected timestamps.'));
        $this->tempStore->delete('selected_timestamps');
      }
    }
    else {
      $values = array_filter($form_state->getValue('timestamps'));
      if (empty($values)) {
        $this->messenger()->addWarning($this->t('No timestamps selected.'));
        return;
      }

      $this->tempStore->set('selected_timestamps', $values);
      $form_state->setRebuild(TRUE);
      $form_state->set('confirm', TRUE);
    }
  }

  /**
   * Cancel the confirmation and reset the form.
   */
  public function cancel(array &$form, FormStateInterface $form_state) {
    $this->tempStore->delete('selected_timestamps');
    $form_state->setRebuild(TRUE);
    $form_state->set('confirm', FALSE);
  }

}
