(function ($, Drupal) {
  Drupal.behaviors.db_health_table_chart = {
    attach: function (context, settings) {
      if (!context.querySelector('#tableSizeChart')) {
        return;
      }

      if (context.chartInitialized) {
        return;
      }

      context.chartInitialized = true;

      const rawData = drupalSettings.db_health.chart_data || [];
      let chart;

      function buildChart(useCount = false) {
        const timestamps = [...new Set(rawData.map(item => item.timestamp))].sort();
        const labels = timestamps.map(ts => new Date(ts * 1000).toISOString().slice(0, 19).replace('T', ' '));

        const tableGroups = {};
        rawData.forEach(({ table_name, size, count, timestamp }) => {
          if (!tableGroups[table_name]) {
            tableGroups[table_name] = {};
          }
          tableGroups[table_name][timestamp] = useCount ? count : size;
        });

        const datasets = Object.entries(tableGroups).map(([tableName, timeMap], index) => {
          const color = `hsl(${index * 30 % 360}, 70%, 50%)`;
          const data = timestamps.map(ts => {
            const value = timeMap[ts] || 0;
            return useCount ? value : value / 1024 / 1024; // count or MB
          });

          return {
            label: tableName,
            data,
            borderColor: color,
            backgroundColor: color,
            fill: false,
            tension: 0.2,
          };
        });

        const title = useCount ? 'Database Table Row Count Over Time' : 'Database Table Sizes Over Time (MB)';
        const yLabel = useCount ? 'Row Count' : 'Size (MB)';

        const config = {
          type: 'line',
          data: { labels, datasets },
          options: {
            responsive: true,
            interaction: {
              mode: 'nearest',
              intersect: false,
            },
            plugins: {
              title: {
                display: true,
                text: title,
              },
              tooltip: {
                callbacks: {
                  label: (ctx) => {
                    const val = ctx.parsed.y;
                    return `${ctx.dataset.label}: ${useCount ? val : val.toFixed(2) + ' MB'}`;
                  }
                }
              },
              legend: {
                display: false,
              },
            },
            scales: {
              x: { title: { display: true, text: 'Timestamp' } },
              y: {
                title: { display: true, text: yLabel },
                beginAtZero: true,
              },
            },
          },
        };

        if (chart) {
          chart.destroy();
        }

        const ctx = document.getElementById('tableSizeChart').getContext('2d');
        chart = new Chart(ctx, config);

        renderLegend(chart);
      }

      function renderLegend(chartInstance) {
        const legendContainer = document.getElementById('chart-legend');
        let html = '<ul class="chart-legend">';
        chartInstance.data.datasets.forEach((dataset, index) => {
          html += `
            <li data-index="${index}" class="legend-item" style="opacity:1; cursor:pointer;">
              <span style="
                background-color:${dataset.borderColor};
                width:12px;
                height:12px;
                display:inline-block;
                margin-right:5px;
              "></span>${dataset.label}
            </li>`;
        });
        html += '</ul>';
        legendContainer.innerHTML = html;
        document.querySelector('.db-health-chart__colummn--legend').classList.add('visible');
        updateLegendCount();

        document.querySelectorAll('.legend-item').forEach(item => {
          item.addEventListener('click', () => {
            const index = parseInt(item.getAttribute('data-index'));
            const meta = chartInstance.getDatasetMeta(index);
            meta.hidden = !meta.hidden;
            chartInstance.update();
            item.style.opacity = meta.hidden ? '0.5' : '1';
            updateLegendCount();
          });
        });
      }

      function updateLegendCount() {
        const visibleCount = Array.from(document.querySelectorAll('.legend-item'))
          .filter(item => item.style.opacity !== '0.5').length;
        document.getElementById('legend-count').textContent = `Visible tables: ${visibleCount}`;
      }

      // Control buttons
      document.getElementById('hide-all').addEventListener('click', () => {
        chart.data.datasets.forEach((_, i) => {
          chart.getDatasetMeta(i).hidden = true;
          document.querySelector(`.legend-item[data-index="${i}"]`).style.opacity = '0.5';
        });
        chart.update();
        updateLegendCount();
      });

      document.getElementById('show-all').addEventListener('click', () => {
        chart.data.datasets.forEach((_, i) => {
          chart.getDatasetMeta(i).hidden = false;
          document.querySelector(`.legend-item[data-index="${i}"]`).style.opacity = '1';
        });
        chart.update();
        updateLegendCount();
      });

      document.getElementById('toggle-visible').addEventListener('click', () => {
        document.querySelectorAll('.legend-item').forEach(item => {
          if (item.style.display !== 'none') {
            const index = parseInt(item.getAttribute('data-index'));
            const meta = chart.getDatasetMeta(index);
            meta.hidden = !meta.hidden;
            item.style.opacity = meta.hidden ? '0.5' : '1';
          }
        });
        chart.update();
        updateLegendCount();
      });

      document.getElementById('legend-filter').addEventListener('input', function () {
        const filter = this.value.toLowerCase();
        document.querySelectorAll('.legend-item').forEach(item => {
          const label = item.textContent.toLowerCase();
          item.style.display = label.includes(filter) ? 'flex' : 'none';
        });
        updateLegendCount();
      });

      // Rebuild chart with count instead of size
      document.getElementById('rebuild-with-count').addEventListener('click', () => {
        buildChart(true);
      });
      document.getElementById('rebuild-with-size').addEventListener('click', () => {
        buildChart(false);
      });

      // Initial chart build
      buildChart(false);
    }
  };
})(jQuery, Drupal);
