<?php

namespace Drupal\db_health;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;

/**
 * Checks and stores paragraph table sizes.
 */
class DbHealthSizeChecker {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new \Drupal\db_health\DbHealthSizeChecker object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(Connection $database, ConfigFactoryInterface $config_factory) {
    $this->database = $database;
    $this->configFactory = $config_factory;
  }

  /**
   * Runs the checker and stores sizes.
   */
  public function run(): void {
    $connection = \Drupal::database();
    $driver = $connection->driver();
    $tables_arr = [];
    $timestamp = time();

    if ($driver === 'mysql') {
      $tables = $connection->query("SHOW TABLE STATUS")->fetchAllAssoc('Name');
    }
    elseif ($driver === 'pgsql') {
      $tables = [];
      $result = $connection->query("
        SELECT
          c.relname AS Name,
          pg_total_relation_size(c.oid) AS total_bytes,
          COALESCE(s.n_live_tup, 0) AS row_count
        FROM pg_class c
        LEFT JOIN pg_namespace n ON n.oid = c.relnamespace
        LEFT JOIN pg_stat_user_tables s ON s.relname = c.relname
        WHERE c.relkind = 'r'
          AND n.nspname = 'public'
      ");
      foreach ($result as $row) {
        $tables[$row->name] = (object) [
          'Data_length' => (int) $row->total_bytes,
          'Index_length' => 0,
          'Name' => $row->name,
          'Row_count' => (int) $row->row_count,
        ];
      }
    }
    else {
      throw new \Exception('Unsupported database driver: ' . $driver);
    }

    $tables_used = $this->getTables();

    foreach ($tables as $name => $info) {
      if ($driver === 'mysql') {
        $size = (int) $info->Data_length + (int) $info->Index_length;
        $count = (int) $connection->query("SELECT COUNT(*) FROM {{$name}}")->fetchField();
      }
      elseif ($driver === 'pgsql') {
        $size = (int) $info->Data_length;
        $count = (int) $info->Row_count;
      }

      if ($name !== 'db_health_sizes' && $name !== 'db_health_table_names' && $name != 'db') {
        $table_id = $this->getOrCreateTableId($name);
        $connection->insert('db_health_sizes')
          ->fields([
            'table_id' => $table_id,
            'size' => $size,
            'count' => $count,
            'timestamp' => $timestamp,
          ])->execute();

        $tables_arr[] = $name;
      }
    }

    // Handle tables that existed before but are now missing.
    $diff = array_diff($tables_used, $tables_arr);
    if ($diff) {
      foreach ($diff as $table_name) {
        $table_id = $this->getOrCreateTableId($table_name);
        if ($table_name !== 'db_health_sizes' && $table_name !== 'db_health_table_names' && $table_name != 'db') {
          $connection->insert('db_health_sizes')
            ->fields([
              'table_id' => $table_id,
              'size' => 0,
              'count' => 0,
              'timestamp' => $timestamp,
            ])->execute();
        }
      }
    }

    // Log full DB size as a pseudo-table "db".
    if ($driver === 'mysql') {
      $query = $connection->query("
        SELECT ROUND(SUM(data_length + index_length), 2) AS size_mb 
        FROM information_schema.tables 
        WHERE table_schema = :schema",
        [':schema' => $connection->getConnectionOptions()['database']]
      );
      $size = $query->fetchField();

      $total_rows = $connection->query("
        SELECT SUM(TABLE_ROWS)
        FROM information_schema.tables
        WHERE table_schema = :schema",
        [':schema' => $connection->getConnectionOptions()['database']]
      )->fetchField();
    }
    elseif ($driver === 'pgsql') {
      $query = $connection->query("
        SELECT ROUND(SUM(pg_total_relation_size(c.oid))::numeric / 1048576, 2) AS size_mb
        FROM pg_class c
        LEFT JOIN pg_namespace n ON n.oid = c.relnamespace
        WHERE c.relkind = 'r' AND n.nspname = 'public'
      ");
      $size = $query->fetchField();

      $total_rows = $connection->query("
        SELECT SUM(s.n_live_tup)
        FROM pg_stat_user_tables s
      ")->fetchField();
    }

    $table_id = $this->getOrCreateTableId('db');
    $connection->insert('db_health_sizes')
      ->fields([
        'table_id' => $table_id,
        'size' => $size,
        'count' => $total_rows,
        'timestamp' => $timestamp,
      ])->execute();
  }

  /**
   * Lists all size records with table names.
   *
   * @return array
   *   An array of size records with table names.
   */
  public function list() {
    $query = \Drupal::database()->select('db_health_sizes', 'phs')
      ->fields('phs', ['size', 'timestamp'])
      ->fields('t', ['table_name'])
      ->orderBy('timestamp', 'ASC');
    $query->innerJoin('db_health_table_names', 't', 'phs.table_id = t.id');

    return $query->execute()->fetchAll();
  }

  /**
   * Get distinct paragraph table names used.
   *
   * @return array
   *   An array of table names.
   */
  private function getTables() {
    $query = \Drupal::database()->select('db_health_sizes', 'phs');
    $query->innerJoin('db_health_table_names', 't', 'phs.table_id = t.id');
    $query->addField('t', 'table_name');
    $result = $query->execute()->fetchCol();

    $result = array_unique($result);

    return $result;
  }

  /**
   * Get or create table_id by table name.
   *
   * @param string $table_name
   *   The table name.
   *
   * @return int
   *   The table ID.
   */
  private function getOrCreateTableId(string $table_name): int {
    $id = \Drupal::database()->query("SELECT id FROM {db_health_table_names} WHERE table_name = :name", [
      ':name' => $table_name,
    ])->fetchField();

    if (!$id) {
      $id = \Drupal::database()->insert('db_health_table_names')
        ->fields(['table_name' => $table_name])
        ->execute();
    }

    return (int) $id;
  }

}
