<?php

namespace Drupal\db_health\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for DB Health module.
 */
class DbHealthController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs a new \Drupal\db_health\Controller\DbHealthController object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database')
    );
  }

  /**
   * Show all tables.
   */
  public function list() {
    $query = $this->database->select('db_health_sizes', 's');
    $query->innerJoin('db_health_table_names', 't', 's.table_id = t.id');
    $query->fields('s', ['size', 'timestamp', 'count']);
    $query->fields('t', ['table_name']);
    $query->orderBy('s.timestamp', 'DESC');

    $data = $query->execute()->fetchAll();

    $prepared_data = [];
    foreach ($data as $record) {
      $prepared_data[] = [
        'table_name' => $record->table_name,
        'size' => (int) $record->size,
        'count' => (int) $record->count,
        'timestamp' => (int) $record->timestamp,
      ];
    }

    // Sort by 'size' in descending order.
    usort($prepared_data, function ($a, $b) {
      return $b['size'] <=> $a['size'];
    });

    return [
      '#theme' => 'db_health_table_chart',
      '#chart_data' => json_encode(array_values($prepared_data)),
      '#table_name' => 'All',
      '#attached' => [
        'drupalSettings' => [
          'db_health' => [
            'chart_data' => array_values($prepared_data),
          ],
        ],
        'library' => [
          'db_health/db_health.chart',
        ],
      ],
    ];
  }

  /**
   * Run for specific table.
   */
  public function table($table) {
    $query = $this->database->select('db_health_sizes', 's');
    $query->innerJoin('db_health_table_names', 't', 's.table_id = t.id');
    $query->fields('s', ['size', 'timestamp', 'count']);
    $query->fields('t', ['table_name']);
    $query->orderBy('s.timestamp', 'DESC');
    $query->condition('t.table_name', $table);
    $query->range(0, 100);

    $data = $query->execute()->fetchAll();

    $prepared_data = [];
    foreach ($data as $record) {
      $prepared_data[] = [
        'table_name' => $record->table_name,
        'size' => (int) $record->size,
        'count' => (int) $record->count,
        'timestamp' => (int) $record->timestamp,
      ];
    }

    // Sort by 'size' in descending order.
    usort($prepared_data, function ($a, $b) {
      return $b['size'] <=> $a['size'];
    });

    return [
      '#theme' => 'db_health_table_chart',
      '#chart_data' => json_encode(array_values($prepared_data)),
      '#table_name' => $table,
      '#attached' => [
        'drupalSettings' => [
          'db_health' => [
            'chart_data' => array_values($prepared_data),
          ],
        ],
        'library' => [
          'db_health/db_health.chart',
        ],
      ],
    ];
  }

}
