<?php

namespace Drupal\Tests\dbee\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\user\Entity\User;

/**
 * Base class for the dbee modules tests.
 *
 * Correctly install sub modules, provide emails and a testing function.
 */
abstract class DbeeKernelTestBase extends KernelTestBase {
  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['system', 'config', 'user', 'key', 'encrypt', 'real_aes', 'dbee'];

  /**
   * Number of basic user to create.
   *
   * @var array
   */
  protected $nUsers = 10;

  /**
   * Array keyed by uid providing users mail and int original values.
   *
   * @var array
   */
  protected $usersInfo = [];

  /**
   * Number of users stored in db.
   *
   * @var int
   */
  protected $totalUsers = 0;

  /**
   * Number of users stored in db that should be encrypted (valid email).
   *
   * @var int
   */
  protected $nUpdatedUsers = 0;

  /**
   * Make sure that the dbee functions are availables.
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->installSchema('user', ['users_data']);
    if (PHP_SAPI !== 'cli') {
      $this->markTestSkipped('This test has to be run from the CLI');
    }

    $this->installConfig(['system']);

    // Create many users.
    $this->dbeeCreateManyUsers();
    if (empty($this->totalUsers)) {
      // Set the count of users.
      $this->dbeeGetUsersInfo();
    }
  }

  /**
   * Check that the emails stored in db are the expected ones.
   *
   * Check the 4 values : mail, init, and the sensitive case mail (dbee). Try
   * to decrypt it back.
   *
   * @param array $usersInfo
   *   Multidimensional array storing user mail and init original values
   *   (uncrypted) keyed by the user ID.
   * @param bool $installed
   *   Inform if the dbee module is enabled. TRUE for enabled (datas should be
   *   encrypted, FALSE for disabled (datas should be decrypted). Default is
   *   TRUE.
   *
   * @return bool
   *   TRUE if all test users email addresses are as expected.
   */
  protected function dbeeAllUsersValid(array $usersInfo, $installed = TRUE) {
    // Test all email address.
    $all_succeed = TRUE;
    $dbee_fields = ['mail', 'init'];
    foreach ($usersInfo as $uid => $source) {
      // $source is the real values (uncrypted).
      $storeds = dbee_stored_users($uid);
      $stored = $storeds[$uid];
      foreach ($dbee_fields as $field) {
        // Set from the source if the data should be encrypted or not.
        $encryption_on = ($installed && dbee_email_to_alter($source[$field]));
        // Test if value seems encrypted.
        $decrypted_stored_data = (!$encryption_on) ? $source[$field] : dbee_decrypt($stored[$field]);
        $is_encrypted = (!empty($stored[$field]) && !dbee_email_to_alter($stored[$field]) && $stored[$field] != $source[$field]);
        $expected = ($decrypted_stored_data == $source[$field] && (($encryption_on && $is_encrypted) || (!$encryption_on && !$is_encrypted)));
        if (!$expected) {
          $all_succeed = FALSE;
          $crypted = (($is_encrypted) ? 'encrypted' : 'uncrypted');
          $expected_crypted = (($encryption_on) ? 'encrypted' : 'uncrypted');

          $this->assertTrue(FALSE, "User {$uid} : the stored {$field} ({$crypted} {$decrypted_stored_data}) is not the expected one ({$expected_crypted}, {$source[$field]})");
        }
      }
    }
    return $all_succeed;
  }

  /**
   * Create many users with various values of mail and init properties.
   *
   * Create as $this->nUsers users. Create at least 10 users to have all
   * possibles kind of values of mail and init. In order to create such users,
   * simply set the $this->nUsers into the class that extends the
   * DbeeKernelTestBase class and call this method from the setUp. This function
   * will set $this->usersInfo, $this->totalUsers, $this->nUpdatedUsers,
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function dbeeCreateManyUsers() {
    // First create 10 users with variant email addresses : sensitive case,
    // lowercase, empty and invalid.
    $n_created = 0;
    $provider = '@eXample.com';
    for ($i = 1; $i <= $this->nUsers; $i++) {
      // Create users, sensitive case and lower case.
      $edit = [];
      $edit['name'] = $this->randomMachineName();
      $edit['mail'] = $edit['name'] . $provider;

      // 3 sensitive case email addresses (5 - 2).
      if (ceil($i / 2) == $i / 2) {
        // 4 lowercase email addresses (5-1).
        $edit['mail'] = mb_strtolower($edit['mail']);
      }

      if (ceil($i / 9) == $i / 9) {
        // 1 empty email addresses.
        $edit['mail'] = '';
      }
      if (ceil($i / 5) == $i / 5) {
        // 2 invalid email address.
        $edit['mail'] = 'This-is-an-invalid-email-' . $this->randomMachineName();
        // The second one is lower case.
        if ($i > 5) {
          $edit['mail'] = mb_strtolower($edit['mail']);
        }
      }

      $edit['pass'] = \Drupal::service('password_generator')->generate();
      $edit['status'] = 1;
      $edit['init'] = $edit['mail'];

      $account = User::create($edit);
      $account->save();
      $account_uid = (!empty($account->id()));
      unset($account);
      $this->assertTrue($account_uid, "User creation succeed for email {$edit['mail']}");
      if ($account_uid) {
        $n_created++;
        $this->usersInfo[$account_uid]['mail'] = $edit['mail'];
        $this->usersInfo[$account_uid]['init'] = $edit['init'];
      }
    }

    if ($n_created > 0) {
      $this->assertTrue($n_created == $this->nUsers, "{$this->nUsers} users have been created successfully");
    }
  }

  /**
   * Store infos for all users.
   */
  protected function dbeeGetUsersInfo() {
    $total_users = $no_update_user = 0;
    $all_users = dbee_stored_users();
    foreach ($all_users as $uid => $values) {
      $total_users++;
      $update = FALSE;
      foreach (['mail', 'init'] as $field) {
        $value = (isset($values[$field])) ? $values[$field] : '';
        if ($value && dbee_email_to_alter($value)) {
          // This si a valid email address.
          $update = TRUE;
        }
        // @todo check if the dbee module is not enabled. In the meantime, call
        // this method only if the dbee module is not enabled.
        $this->usersInfo[$uid][$field] = $value;

      }
      if (!$update) {
        $no_update_user++;
      }
    }
    $updated_users = $total_users - $no_update_user;

    // Set the numbers of total and updated users.
    $this->totalUsers = $total_users;
    $this->nUpdatedUsers = $updated_users;
  }

}
