<?php

namespace Drupal\dbee\Hook;

use Drupal\Component\Utility\EmailValidatorInterface;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Config\ConfigImporter;
use Drupal\Core\Database\Query\AlterableInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\Display\EntityViewDisplayInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Hook\Order\Order;
use Drupal\Core\Hook\Order\OrderAfter;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\dbee\Query;
use Drupal\encrypt\Entity\EncryptionProfile;
use Drupal\key\KeyRepository;
use Drupal\user\UserInterface;
use Psr\Log\LoggerInterface;

/**
 * Hooks for the dbee module.
 */
class EntityHooks {

  use StringTranslationTrait;

  public function __construct(
    private readonly LoggerInterface $logger,
    private readonly ModuleHandlerInterface $module_handler,
    private readonly AccountInterface $currentUser,
    private readonly KeyRepository $keyRepository,
    private readonly EmailValidatorInterface $emailValidator,
    private readonly StateInterface $state,
  ) {}

  /**
   * Implements hook_entity_type_alter().
   */
  #[Hook('entity_type_alter')]
  public function entityTypeAlter(array &$entity_types): void {
    if (isset($entity_types['user'])) {
      $entity_types['user']->setClass('Drupal\dbee\Entity\DbeeUser');
    }
  }

  /**
   * Implements hook_entity_type_build().
   *
   * Increase columns length and remove index on the email column.
   */
  #[Hook('entity_type_build')]
  public function entityTypeBuild(array &$entity_types): void {
    if (!empty($entity_types['user'])) {
      $entity_types['user']->setHandlerClass('storage_schema', 'Drupal\\dbee\\DbeeUserStorageSchema');
    }
  }

  /**
   * Implements hook_entity_load().
   *
   * Decrypts the mail and init addresses when loading a user object.
   * hook_entity_load() is called before hook_ENTITY_TYPE_load().
   */
  #[Hook('entity_load', order: Order::First)]
  public function entityLoad(array $entities, $entity_type_id): void {
    if ($entity_type_id == 'user') {
      // Decrypt the email address when a user is loaded.
      // This makes the email address available to the system.
      foreach ($entities as $record) {
        dbee_extract($record);
      }
    }
  }

  /**
   * Implements hook_entity_presave().
   *
   * Encrypt the email address when saving a user account. hook_entity_presave()
   * is called after hook_ENTITY_TYPE_presave().
   */
  #[Hook('entity_presave', order: Order::Last)]
  public function entityPresave(EntityInterface $entity): void {
    $entity_type_id = $entity->getEntityTypeId();
    switch ($entity_type_id) {
      case 'user':
        // Set a variable containing the unencrypted email address values to
        // encrypt.
        $uncrypted = [];
        foreach (['mail', 'init'] as $dbee_field) {
          if (!empty($entity->$dbee_field->value)) {
            $uncrypted[$dbee_field] = $entity->$dbee_field->value;
          }
          else {
            $uncrypted[$dbee_field] = NULL;
          }
        }
        // Encrypt the email address values.
        $to_update = dbee_store($uncrypted);
        // Replace unencrypted email address values with encrypted email address
        // values in the User object.
        foreach ($to_update as $field => $value) {
          // Mail and init.
          if ($value) {
            $entity->$field->value = $value;
          }
        }
        break;
    }
  }

  /**
   * Implements hook_ENTITY_TYPE_insert() and hook_ENTITY_TYPE_update().
   *
   * Decrypt the email addresses after saving.
   */
  #[Hook('user_insert', order: Order::First)]
  #[Hook('user_update', order: Order::First)]
  public function userInsertUpdate(UserInterface $entity) {
    // Decrypt the email addresses after saving.
    // The user object provides unencrypted email address values.
    dbee_extract($entity);
    // Reset cache on the new/updated entity.
    _dbee_all_users_uncrypted(FALSE, $entity);
  }

  /**
   * Implements hook_entity_extra_field_info().
   *
   * View extra field for encryption status.
   */
  #[Hook('entity_extra_field_info')]
  public function entityExtraFieldInfo() {
    $fields['user']['user']['display']['dbee'] = [
      'label' => $this->t('Email encryption'),
      'description' => $this->t('Display the email encryption status.'),
      'weight' => 15,
      // member_for weight is 5. dbee will appear below the member_for markup.
      'visible' => FALSE,
    ];

    return $fields;
  }

  /**
   * Implements hook_ENTITY_TYPE_view().
   *
   * This function displays an explicit message on the user profile page
   * about email address encryption. It is displayed only to users with the
   * 'administer dbee' permission.
   */
  #[Hook('user_view')]
  public function userView(array &$build, UserInterface $entity, EntityViewDisplayInterface $display, $view_mode): void {
    if ($display->getComponent('dbee')) {
      // Use a static query to bypass the dbee_query_alter() function.
      $analyze = dbee_analyze_users($entity);
      if (!empty($analyze)) {
        // Display a new field on the user profile page. Use the same theming as
        // 'member_for' from user_user_view().
        $build['dbee'] = [
          '#type' => 'item',
          '#markup' => '<h4 class="label">' . $this->t('Security') . '</h4> ' . $analyze['message'],
          '#access' => $this->currentUser->hasPermission('administer dbee'),
        ];
      }
    }
  }

  /**
   * Implements hook_query_alter().
   *
   * Handle decrypting process on tagged queries.
   */
  #[Hook('query_alter', order: Order::Last)]
  public function queryAlter(AlterableInterface $query) {
    $class = new Query();
    $class->dbee($query);
  }

  /**
   * Implements hook_query_user_load_multiple_alter().
   */
  #[Hook('query_user_load_multiple_alter', order: Order::Last)]
  public function queryUserLoadMultipleAlter(AlterableInterface $query) {}

  /**
   * Implements hook_query_pager_alter().
   */
  #[Hook('query_pager_alter', order: Order::Last)]
  public function queryPagerAlter(AlterableInterface $query) {}

  /**
   * Implements hook_entity_query_alter().
   */
  #[Hook('entity_query_alter', order: Order::Last)]
  public function entityQueryAlter(AlterableInterface $query) {}

  /**
   * Implements hook_ENTITY_TYPE_presave().
   *
   * Target the dbee key and encryption profile for creation and changes.
   */
  #[Hook('key_presave')]
  #[Hook('encryption_profile_presave')]
  public function entityKeyEncryptionProfilePresave(EntityInterface $entity) {
    $entity_type_id = $entity->getEntityTypeId();
    if (($entity_type_id == 'key' && $entity->getOriginalId() == dbee_current_key_id()) || ($entity_type_id == 'encryption_profile' && $entity->getOriginalId() == DBEE_ENCRYPT_NAME)) {
      if (!$entity->isNew() && dbee_entity_change($entity)) {
        // The dbee key will change.
        $set_prev_encrypt = FALSE;
        $set_prev_key = FALSE;
        $reencrypt_all = FALSE;
        if ($entity_type_id == 'key') {
          if (!$this->keyRepository->getKey(DBEE_PREV_KEY_NAME)) {
            $copy = $entity->original->createDuplicate();
            $copy->set('name', DBEE_PREV_KEY_NAME);
            $copy->set('id', DBEE_PREV_KEY_NAME);
            $copy->set('label', 'Previous DataBase Email Encryption key');
            $copy->set('description', 'Previous Dbee key for decrypting users email addresses.');
            if ($copy->save() && $this->keyRepository->getKey(DBEE_PREV_KEY_NAME)) {
              $this->logger->info('Dbee key is going to change, previous key is saved.');
              $set_prev_encrypt = $set_prev_key = TRUE;
            }
            else {
              $this->logger->critical('Dbee key is going to change, and the previous key is not saved !');
            }
          }
          else {
            // Do not overwrite previous key.
            $set_prev_encrypt = TRUE;
          }
        }
        elseif ($entity_type_id == 'encryption_profile') {
          $set_prev_encrypt = TRUE;
        }

        if ($set_prev_encrypt) {
          $dbee_encrypt = ($entity_type_id != 'encryption_profile') ? EncryptionProfile::load(DBEE_ENCRYPT_NAME) : $entity->original;
          $prev_encrypt_profile = EncryptionProfile::load(DBEE_PREV_ENCRYPT_NAME);
          if ($dbee_encrypt && !$prev_encrypt_profile) {
            $copy = $dbee_encrypt->createDuplicate();
            $copy->set('name', DBEE_PREV_ENCRYPT_NAME);
            $copy->set('id', DBEE_PREV_ENCRYPT_NAME);
            $copy->set('label', "Previous Dbee {$dbee_encrypt->getEncryptionMethodId()}");
            if ($set_prev_key) {
              $copy->set('encryption_key', DBEE_PREV_KEY_NAME);
            }
            if ($copy->save() && EncryptionProfile::load(DBEE_PREV_ENCRYPT_NAME)) {
              $this->logger->info('Dbee key is going to change, encrypt profile saved with prev key values.');
              // Save the new settings.
              // Next, decrypt all user email addresses and finally, re-encrypt
              // the email addresses with the new parameters.
              $reencrypt_all = TRUE;
            }
            else {
              $this->logger->critical('Dbee key is going to change, encrypt profile is not saved with prev key values !');
            }
          }
          elseif ($dbee_encrypt && $prev_encrypt_profile) {
            $reencrypt_all = TRUE;
          }
        }

        if ($reencrypt_all) {
          // Wait for the save to complete. Flag to re-encrypt.
          $entity->dbee_change = TRUE;
        }
      }
    }
  }

  /**
   * Implements hook_ENTITY_TYPE_insert() and hook_ENTITY_TYPE_update().
   *
   * Target the dbee key and encryption profile for creation and changes.
   */
  #[Hook('key_insert')]
  #[Hook('key_update')]
  #[Hook('encryption_profile_insert')]
  #[Hook('encryption_profile_update')]
  public function entityKeyEncryptionProfileInsertUpdate(EntityInterface $entity) {
    if ($entity->getEntityTypeId() == 'key' && $entity->getOriginalId() == dbee_current_key_id() || ($entity->getEntityTypeId() == 'encryption_profile' && $entity->getOriginalId() == DBEE_ENCRYPT_NAME)) {
      $config_install = $this->state->get('dbee.install_from_config');
      $config_changes = $this->state->get('dbee.changes_from_config');
      $encrypt_profile = ($entity->getEntityTypeId() == 'encryption_profile') ? $entity : EncryptionProfile::load(DBEE_ENCRYPT_NAME);
      if (is_null($config_install) && is_null($config_changes)) {
        if (!empty($entity->dbee_change)) {
          // Batch for decrypting all user email addresses and then
          // re-encrypting them with the new parameters.
          if ($encrypt_profile) {
            $this->module_handler->loadInclude('dbee', 'inc', 'dbee.users');
            if (!$entity->isNew()) {
              dbee_update_crypt_all('change', 'change', TRUE);
              dbee_update_crypt_all('encrypt', 'change', TRUE);
            }
            else {
              dbee_update_crypt_all('encrypt');
            }
          }
        }
      }
      elseif ($config_install) {
        // Fire form config import on install.
        if (($entity->getEntityTypeId() == 'key' && !isset($config_install['encryption_profile']) && $encrypt_profile) || $entity->getEntityTypeId() == 'encryption_profile') {
          // The dbee key has been set via config import but we still waiting
          // for the dbee encryption_profile to be set. Wait.
          $this->module_handler->loadInclude('dbee', 'install');
          _dbee_install_process();
          $this->state->delete('dbee.install_from_config');
        }
      }
      elseif ($config_changes) {
        // It will be handle in _dbee_config_importer_changes_step() batch.
      }
    }
  }

  /**
   * Implements hook_ENTITY_TYPE_access().
   *
   * Prevent deleting dbee encrypt and key entities.
   * Use the dbee module permission for updating.
   */
  #[Hook('key_access')]
  #[Hook('encryption_profile_access')]
  public function entityKeyEncryptionProfileAccess(EntityInterface $entity, $operation, AccountInterface $account): AccessResultInterface {
    if (in_array($entity->getEntityTypeId(), ['encryption_profile', 'key'])) {
      $id = $entity->getOriginalId();
      $protected_encrypt_ids = [DBEE_ENCRYPT_NAME, DBEE_PREV_ENCRYPT_NAME];
      $protected_key_ids = [DBEE_DEFAULT_KEY_NAME, DBEE_PREV_KEY_NAME];
      $dbee_current_key_id = dbee_current_key_id();
      if (!in_array($dbee_current_key_id, $protected_key_ids)) {
        $protected_key_ids[] = $dbee_current_key_id;
      }
      if (($entity->getEntityTypeId() == 'encryption_profile' && in_array($id, $protected_encrypt_ids)) || ($entity->getEntityTypeId() == 'key' && in_array($id, $protected_key_ids))) {
        return match($operation) {
          'delete' => AccessResult::forbidden(),
          'view' => AccessResult::allowedIfHasPermission($account, 'administer dbee'),
          'update' => ((in_array($id, [DBEE_PREV_KEY_NAME, DBEE_PREV_KEY_NAME])) ? AccessResult::forbidden() : AccessResult::allowedIfHasPermission($account, 'administer dbee')),
          default => AccessResult::neutral(),
        };
      }
    }
    return AccessResult::neutral();
  }

  /**
   * Implements hook_config_import_steps_alter().
   *
   * Note : When installing dbee module via config import, this hook is not
   * fire. We use the dbee_install() hook instead.
   */
  #[Hook('config_import_steps_alter')]
  public function configImportStepsAlter(&$sync_steps, ConfigImporter $config_importer) {
    $modules = 'core.extension';
    $search = [
      'key.key.' . dbee_current_key_id() => 'key',
      'encrypt.profile.' . DBEE_ENCRYPT_NAME => 'encryption_profile',
      $modules => 'module',
    ];
    $dbee_configs = [];
    foreach (['delete', 'create', 'rename', 'update'] as $op) {
      $changes = $config_importer->getUnprocessedConfiguration($op);
      if (!empty($changes)) {
        foreach ($changes as $config_name) {
          if (isset($search[$config_name])) {
            if ($config_name != $modules) {
              $dbee_configs[$search[$config_name]][$op] = $op;
            }
            else {
              // Look for dbee module in core.extension. From container.
              $dbee_configs_container = dbee_changes_form_config_import();
              if (isset($dbee_configs_container['module'])) {
                foreach ($dbee_configs_container['module'] as $op_module) {
                  $dbee_configs[$search[$config_name]][$op_module] = $op_module;
                }
              }
            }
          }
        }
      }
    }
    if (!empty($dbee_configs)) {
      // If deleting the dbee module, decrypting is successfully handled via
      // dbee_uninstall() and the ConfigImporterChanges() won't be called,
      // leading to fatal error.
      if (empty($dbee_configs['module']['delete'])) {
        $this->state->set('dbee.changes_from_config', $dbee_configs);
        $sync_steps[] = '_dbee_config_importer_changes_step';
      }
      else {
        // Set a flag informing of the uninstall.
        $this->state->set('dbee.uninstall_from_config', $dbee_configs);
      }
    }
  }

  /**
 * Implements hook_help().
 */
  #[Hook('help')]
  public function help($route_name, RouteMatchInterface $route_match): string|\Stringable|array|null {
    switch ($route_name) {
      case 'help.page.dbee':
        $output = '';
        $output .= '<h2>' . $this->t('About') . '</h2>';
        $output .= '<p>' . $this->t('The DataBaseEmail Encryption (dbee) module protects users email address, encrypting them into the database.') . '</p>';
        $output .= '<p>' . $this->t('This module does not alter user experience.') . '</p>';
        $output .= '<p>' . $this->t('In case of database hacking, this sensitive data would be useless for the hacker.') . '</p>';
        $output .= '<h2>' . $this->t('Uses') . '</h2>';
        $output .= '<dl>';
        $output .= '<dt>' . $this->t('Configuring encryption') . '</dt>';
        $output .= '<dd>' . $this->t('Dbee <a href=":key">Encryption <em>key</em> page</a> and <a href=":profile">Encryption <em>Profile</em> page</a>', [
          ':key' => Url::fromRoute('entity.key.collection')->toString(),
          ':profile' => Url::fromRoute('entity.encryption_profile.collection')->toString(),
        ]) . '</dd>';
        $output .= '<dt>' . $this->t('Are the email addresses are encrypted ?') . '</dt>';
        $output .= '<dd>' . $this->t('Encryption is fired when the module is installed and emails are decrypted when the module is uninstalled. You can check the encryption status on <a href=":status">the <em>Status Report</em> page</a>.', [':status' => Url::fromRoute('system.status')->toString()]) . '</dd>';
        $output .= '<dt>' . $this->t('Advice') . '</dt>';
        $output .= '<dd>' . $this->t('During module install, the <a href=":key">Dbee encryption key</a> is generated. For security reason, you should store the encryption key <em>outside the database</em>, as an example into a file outside the webroot.', [':key' => Url::fromRoute('entity.key.edit_form', ['key' => DBEE_ENCRYPT_NAME])->toString()]) . '</dd>';
        $output .= '<dt>' . $this->t('Warning') . '</dt>';
        $output .= '<dd>' . $this->t('<strong>keep a copy of your encryption key into a secure location</strong> : loosing this key would definitively prevent access to all your users email addresses !') . '</dd>';
        $output .= '</dl>';
        return $output;
    }
    return NULL;
  }

  /**
   * Implements hook_tokens().
   *
   * Fix compatibility with the webform module. Alter the
   * Drupal\webform\Hook\WebformTokensHooks::tokens() hook, which use
   * select('users_field_data', 'u')[...]->execute()->fetchCol(), bypassing
   * decrypting via entity_load() and hook_query_alter(). Used for Webform roles
   * mail token. Fired from
   * Drupal\webform\Plugin\WebformHandler\EmailWebformHandler::buildElement().
   * The token [webform_role:authenticated] should return an array of encrypted
   * mails. They have to be decrypted by the dbee_tokens() hook.
   */
  #[Hook('tokens', order: new OrderAfter(['webform']))]
  public function dbeeTokens($type, $tokens, array $data, array $options, BubbleableMetadata $bubbleable_metadata) {
    if (!$this->module_handler->moduleExists('webform') || !$this->module_handler->moduleExists('token')) {
      return [];
    }

    $replacements = [];
    if ($type == 'webform_role' && !empty($data['webform_role'])) {
      $replacements_crypted = webform_tokens($type, $tokens, $data, $options, $bubbleable_metadata);
      foreach ($replacements_crypted as $original => $crypted_str) {
        $crypted_arr = explode(',', $crypted_str);
        $decrypted = [];
        foreach ($crypted_arr as $crypted) {
          $decrypted[] = dbee_decrypt($crypted);
        }
        $replacements[$original] = implode(',', $decrypted);
      }
    }
    return $replacements;
  }

}
