<?php

namespace Drupal\dbee\Commands;

use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Database\Connection;
use Drupal\Core\Messenger\MessengerInterface;
use Drush\Utils\StringUtils;

/**
 * A drush command for users emails correct decrypt verifying.
 */
class DbeeCommands extends DrushCommands {

  /**
   * An amount of the records to be processed.
   */
  const BATCH_STEP = 1000;

  /**
   * Current database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $connection;

  /**
   * Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Construct drush command.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   Current database connection.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   Messenger service.
   */
  public function __construct(Connection $connection, MessengerInterface $messenger) {
    parent::__construct();
    $this->connection = $connection;
    $this->messenger = $messenger;
  }

  /**
   * The logic to declare that encryption has failed.
   *
   * Set fail when the account has data in the mail column and:
   * - it is valid mail and should be encrypted but is not.
   * - it is not email : it may be an invalid email address value or a corrupted
   * encrypted email (hard to say, length can help).
   * Empty mail values and correctly decrypted emails will return FALSE.
   *
   * @param array $analyze
   *   Formated analyze from the dbee_analyze_users() function.
   *
   * @return array|true|false:
   *   Array of invalids uids ou TRUE if fails with no uids or FALSE if succeed.
   */
  protected static function emailFails($analyze) {
    if (empty($analyze['count'])) {
      // Fail, it should be provided.
      return TRUE;
    }
    else {
      $counts = $analyze['count'];
      $invalids = ['not_email', 'corrupted', 'not_crypted'];
      foreach ($invalids as $status) {
        if (!empty($counts[$status])) {
          return (!empty($analyze['invalid_uids'])) ? $analyze['invalid_uids'] : TRUE;
        }
      }
    }
    // All emails succeed.
    return FALSE;
  }

  /**
   * Verify email decryption for all users.
   */
  #[CLI\Command(name: 'dbee:verify-users-decrypt-all', aliases: ['dbee-verify-all'])]
  #[CLI\Usage(name: 'dbee:verify-users-decrypt-all', description: 'Verify email decryption for all users')]
  public function verifyAllEmailsDecryption() {
    $max = $this->connection->select('users_field_data', 'ufd')
      ->fields('ufd', ['uid'])
      ->condition('ufd.uid', 0, '!=')
      ->countQuery()->execute()->fetchField();

    $step = 0;
    $batch_builder = new BatchBuilder();
    $batch_builder->setFinishCallback([static::class, 'verifyEmailsFinish']);
    do {
      $batch_builder->addOperation(
        [static::class, 'verifyEmailsByRange'],
        [$step, $step + static::BATCH_STEP]
            );
      $step += static::BATCH_STEP;
    } while ($step < $max);
    batch_set($batch_builder->toArray());
    drush_backend_batch_process();
  }

  /**
   * Verify email decryption for certain users.
   *
   * @param array $uids
   *   Options array.
   */
  #[CLI\Command(name: 'dbee:verify-users-decrypt', aliases: ['dbee-verify'])]
  #[CLI\Argument(name: 'uids', description: 'A comma-separated list of user ids to be verified')]
  #[CLI\Usage(name: 'dbee:verify-users-decrypt', description: 'Verify email decryption for certain users')]
  public function verifyEmailsDecryption($uids = []) {
    $uids = StringUtils::csvToArray($uids);
    if (empty($uids)) {
      $this->messenger->addWarning('Please, specify a users to be verified by --uid option.');
      return;
    }
    $batch_builder = new BatchBuilder();
    foreach (array_chunk($uids, static::BATCH_STEP) as $chunk) {
      $batch_builder->addOperation(
        [static::class, 'verifyEmails'],
        [$chunk]
      );
    }
    $batch_builder->setFinishCallback([static::class, 'verifyEmailsFinish']);
    batch_set($batch_builder->toArray());
    drush_backend_batch_process();
  }

  /**
   * Email decrypt verification batch callback, by users range.
   *
   * @param int $start
   *   The starting range position.
   * @param int $range
   *   The number of records to return from the result set.
   * @param mixed $context
   *   The batch API context.
   */
  public static function verifyEmailsByRange(int $start, int $range, mixed &$context = []) {
    /** @var \Drupal\Component\Utility\EmailValidatorInterface $validator */
    $analyze = dbee_analyze_users(NULL, $start, $range);
    $fail = self::emailFails($analyze);
    if (!empty($fail)) {
      if (is_array($fail)) {
        foreach ($fail as $uid) {
          $context['results']['failed'][] = $uid;
        }
      }
    }
    if (!empty($analyze['message'])) {
      $context['message'] = $analyze['message'];
    }
  }

  /**
   * Email decrypt verification batch callback, by ids.
   *
   * @param array $uids
   *   List of user entity ids to be verified.
   * @param mixed $context
   *   The batch API context.
   */
  public static function verifyEmails(array $uids, mixed &$context = []) {
    foreach ($uids as $uid) {
      $analyze = dbee_analyze_users($uid);
      $fail = self::emailFails($analyze);
      if (!empty($fail)) {
        if (is_array($fail)) {
          foreach ($fail as $uid_fail) {
            $context['results']['failed'][] = $uid_fail;
          }
        }
      }
      if (!empty($analyze['message'])) {
        $context['message'] = $analyze['message'];
      }
    }
  }

  /**
   * Email decrypt verification batch finish callback.
   *
   * @param bool $success
   *   A boolean indicating whether the batch has completed successfully.
   * @param array $results
   *   The value set in $context['results'].
   */
  public static function verifyEmailsFinish(bool $success, array $results) {
    $messenger = \Drupal::messenger();
    if (!$success) {
      $messenger->addError(t('Something went wrong!'));
      return;
    }

    $failed = $results['failed'] ?? [];
    if (empty($failed)) {
      $messenger->addStatus(t('Users have correctly encrypted emails.'));
      return;
    }
    $failed = array_unique($failed);

    $messenger->addWarning(t('Found @count users with an emails that cannot be decrypted or are uncrypted.', [
      '@count' => count($failed),
    ]));
    foreach (array_chunk($failed, static::BATCH_STEP) as $ids) {
      $messenger->addWarning(implode(', ', $ids));
    }
  }

}
