<?php

namespace Drupal\dbee;

use Drupal\Core\Database\Query\AlterableInterface;

/**
 * Handle decrypting process on tagged queries.
 *
 * Encrypt the email addresses on queries. This hook is only called on dynamic
 * tagged queries. The 'dbee' tag is not required. For example, the query from
 * the user_load_by_mail() function is altered because of the
 * 'user_load_multiple' tag. On the contrary, the
 * Drupal\user\Authentication\Provider\Cookie::getUserFromSession() is not
 * tagged, and the result is bypassing entity_load(), that why I had to replace
 * this function by the custom class DbeeCookie().
 * The dbee_query_alter() function encrypts any dynamic tagged query that
 * contains the mail or init field in the where clause.
 * Acting on tags:
 * - If the query contains the tag 'dbee_disabled', no alteration occurs.
 * - If the query contains the tags 'dbee_mail_disable_insensitive_case' or
 *   'dbee_init_disable_insensitive_case': the search will be case-sensitive
 *   (it is not the default behavior on mysql; read the code and issue below).
 * Tags added by the dbee module:
 * - 'dbee_mail' if the query contains users.mail field in the where clause.
 * - 'dbee_init' if the query contains users.init field in the where clause.
 */
class Query {

  /**
   * The current database connection from the query.
   *
   * @var \Drupal\Core\Database\Connection
   */
  public $connection;

  /**
   * Executed from hook_query_alter()
   */
  public function dbee(AlterableInterface $query) {
    $this->queryAlter($query);
  }

  /**
   * Identify encrypted fields/tables on the query.
   *
   * If the query involves encrypted field(s): save the connection in the class
   * and fire whereClause() method for altering the query.
   */
  public function queryAlter(AlterableInterface $query) {

    $users_alias = FALSE;
    $tables = &$query->getTables();
    foreach ($tables as $table_properties) {
      if ($table_properties['table'] == 'users_field_data') {
        // The users table is queried.
        $users_alias = $table_properties['alias'];
        break;
      }
    }

    if ($users_alias) {
      // The {users_field_data} table is queried.
      $dbee_fields = ['mail', 'init'];
      foreach ($dbee_fields as $dbee_field) {
        $field_selected[$dbee_field] = FALSE;
        // Set a default mail alias.
        $dbee_alias[$dbee_field] = $dbee_field;
      }
      // Check if all fields have been loaded.
      if (is_array($tables[$users_alias]) && array_key_exists('all_fields', $tables[$users_alias]) && $tables[$users_alias]['all_fields']) {
        foreach ($dbee_fields as $dbee_field) {
          $field_selected[$dbee_field] = TRUE;
          // Optional. The mail alias is the same as default but the value has
          // been confirmed.
          $dbee_alias[$dbee_field] = $dbee_field;
        }
      }
      else {
        // Or at least the mail or init fields.
        $fields = &$query->getFields();
        foreach ($fields as $field_properties) {
          if ($field_properties['table'] == $users_alias) {
            foreach ($dbee_fields as $dbee_field) {
              if ($field_properties['field'] == $dbee_field) {
                $field_selected[$dbee_field] = TRUE;
                $dbee_alias[$dbee_field] = $field_properties['alias'];
              }
            }
          }
        }
      }

      foreach ($dbee_fields as $dbee_field) {
        if ($field_selected[$dbee_field]) {
          // The 'mail' or the 'init' field from the table 'users_field_data' is
          // queried. The 'dbee_mail' or 'dbee_init' tags inform that the result
          // set will probably return encrypted mail values and should go
          // through the dbee_decrypt() function!
          $query->addTag('dbee_' . $dbee_field);
        }
      }

      // Handle the WHERE clause.
      // The 'dbee_disabled' tag disables all dbee stuff on queries.
      if (!$query->hasTag('dbee_disabled')) {
        $this->connection = $query->getConnection();
        $this->whereClause($query, $users_alias, $dbee_alias);
      }
    }
  }

  /**
   * Helper for reformatting encryption-related database queries.
   *
   * Regarding the field, operator and value in the where clause, return the
   * corresponding changed values. For example replace "WHERE mail LIKE
   * 'john@example.com'" by "WHERE uid IN (2, 7)" or "WHERE uid = 3" or
   * "WHERE 0 = 1" if no match.
   *
   * @param object $query
   *   $query can be Drupal\Core\Database\Query\AlterableInterface or
   *    Drupal\Core\Database\Query\Condition, from self calling (recursion).
   * @param string $users_alias
   *   The table name alias.
   * @param array $dbee_alias
   *   The columns concerned: key is the column name and value is its alias.
   */
  public function whereClause(&$query, $users_alias, $dbee_alias) {
    // Handle the WHERE clause.
    $dbee_fields = ['mail', 'init'];
    $where = &$query->conditions();

    foreach ($where as $placeholder => $where_properties) {
      if (is_array($where_properties) && array_key_exists('field', $where_properties) && array_key_exists('value', $where_properties)) {
        if (is_object($where_properties['field'])) {
          // For nested conditions (db_or()). Used by user_search_execute().
          $this->whereClause($where[$placeholder]['field'], $users_alias, $dbee_alias);
        }
        elseif (is_string($where_properties['field'])) {
          // Used by the load_by_mail_user() core function.
          foreach ($dbee_fields as $dbee_field) {
            // Alias: 'users_field_data.mail' or 'mail' in most cases.
            $where_alias = $this->buildAlias($dbee_field, 'users_field_data', $dbee_alias[$dbee_field], $users_alias);
            $where_dbee_field = $db_funct = FALSE;
            if (in_array($where_properties['field'], $where_alias)) {
              $where_dbee_field = TRUE;
            }
            elseif ($db_funct = $this->dbFunctionsClause($where_properties['field'])) {
              $funct_op = $db_funct['op'];
              foreach ($db_funct['fields'] as $funct_field) {
                if (in_array($funct_field, $where_alias)) {
                  $where_dbee_field = TRUE;
                  break;
                }
              }
            }
            if ($where_dbee_field) {
              // The where clause does contain the mail or init fields!
              // First, handle simple or multiple values.
              $where_strings = [];
              if (is_string($where_properties['value'])) {
                // The user_load_by_mail() core function needs the code below.
                $where_strings[] = $where_properties['value'];
              }
              elseif (is_array($where_properties['value'])) {
                $where_strings = $where_properties['value'];
              }
              else {
                // This case should never occur.
                continue;
              }

              if (count($where_strings) == 1 && current($where_strings) === '') {
                continue;
              }

              // initialization:
              $need_rewrite = FALSE;
              $new_values = [];
              $new_field = FALSE;
              $operator = isset($where_properties['operator']) ? mb_strtoupper($where_properties['operator']) : '';
              if (empty($operator) && !empty($funct_op)) {
                $operator = $funct_op;
              }
              $where_match = in_array($operator, ['LIKE', '=', 'IN']);

              $insensitive_search = (in_array($operator, ['LIKE', 'NOT LIKE']));
              // Special behavior regarding issues #2324701 and #2616264
              // (https://www.drupal.org/node/2616264#comment-10576582)
              // Due to MySQL collation, queries are executed as
              // case-insensitive.
              // Current issue for D8: #2490294
              // Provide a tag to disable this behavior, which is needed for
              // testing and custom modules: dbee_mail_disable_insensitive_case
              // and dbee_init_disable_insensitive_case.
              if (empty($query->alterTags["dbee_{$dbee_field}_disable_insensitive_case"]) && !($this->connection->databaseType() == 'pgsql' && $dbee_field == 'init')) {
                $insensitive_search = 'TRUE';
              }

              // The WHERE clause must decrypt all users in complex searches
              // with wildcards ('_' and '%').
              // Look for unescaped wildcards.
              // Find active '%' and '_'mysql wildcards.
              $pattern_wildcards = '[^\\\\]%|^%|[^\\\\]_|^_';
              // The custom method is not the default one. It is faster but less
              // powerful because it will return the user only if the complete
              // mail string is provided.
              $wildcard_search = FALSE;
              if (in_array($operator, ['LIKE', 'NOT LIKE'])) {
                foreach ($where_strings as $string) {
                  if (preg_match("!$pattern_wildcards!", $string) === 1) {
                    $wildcard_search = TRUE;
                    break;
                  }
                }
              }
              $no_encrypt_value = [];
              if (!$wildcard_search) {
                // Check if the value needs to be encrypted.
                foreach ($where_strings as $string) {
                  $no_encrypt_value[] = (!dbee_email_to_alter($string));
                }
                $no_encrypt_value[] = array_values(array_unique($no_encrypt_value));
                if (count($no_encrypt_value) == 1 && $no_encrypt_value[0]) {
                  $need_rewrite = FALSE;
                }
              }

              if ($wildcard_search || TRUE) {
                $need_rewrite = TRUE;
                $target_field = (!$db_funct) ? $dbee_field : $where_properties['field'];
                $result_uids = $this->whereMailFields2Uid($where_strings, $target_field, $operator, $insensitive_search);
                $n_matching_user = count($result_uids);
                if ($n_matching_user == 0) {
                  // No users match. Returns "WHERE 0 = '1'".
                  $new_field = 1;
                  $new_values[0] = 0;
                }
                else {
                  $new_field = $users_alias . '.uid';
                  $new_values = $result_uids;
                }
              }

              if ($need_rewrite) {
                if (!$where_match) {
                  // Add to ignore empty values.
                  // Retrieve empty values.
                  $empty_users = $this->connection->select('users_field_data', 'ufd')
                    ->fields('ufd', ['uid'])
                    ->condition(($this->connection->condition('OR'))->condition("ufd.{$dbee_field}", '', '=')->isNull("ufd.{$dbee_field}"))
                    ->execute();
                  $added_empty = FALSE;
                  foreach ($empty_users as $account) {
                    if (array_search($account->uid, $new_values) === FALSE) {
                      if ($new_field == $users_alias . '.uid') {
                        $new_values[] = $account->uid;
                        $added_empty = TRUE;
                      }
                      else {
                        // Case 1 = 0 but need to eliminate some results.
                        $new_field = $users_alias . '.uid';
                        $new_values[] = $account->uid;
                      }
                    }
                  }
                  if ($added_empty) {
                    sort($new_values);
                  }
                }

                // Finally, edit the WHERE condition.
                if (count($new_values) == 1) {
                  $where[$placeholder]['operator'] = ($where_match) ? '=' : '<>';
                  $where[$placeholder]['value'] = $new_values[0];
                }
                else {
                  $where[$placeholder]['operator'] = ($where_match) ? 'IN' : 'NOT IN';
                  $where[$placeholder]['value'] = $new_values;
                }
                if (!empty($new_field)) {
                  if ($new_field !== 1) {
                    $where[$placeholder]['field'] = $new_field;
                  }
                  else {
                    // The condition always fails.
                    // Use "Where uid < 0" instead of "Where 1 = 0".
                    $where[$placeholder]['field'] = $users_alias . '.uid';
                    $where[$placeholder]['operator'] = '<';
                    $where[$placeholder]['value'] = 0;
                  }
                }
              }
            }
          }
        }
      }
    }
  }

  /**
   * Provides the corresponding UID to a db WHERE clause, decrypting all users.
   *
   * Convert database wildcards to php regex patterns. Use the cache if called
   * several times during a script.
   *
   * @param array $keys
   *   An array of research strings using database wildcards ('%' and '_').
   * @param string $field
   *   The field in the {users_field_data} table: Can be 'mail' (default)
   *   or 'init'.
   * @param string $operator
   *   The operator on the WHERE clause. Defaults to 'LIKE'.
   * @param bool $insensitive_search
   *   Determines whether the search should be case-insensitive.
   *   Defaults to FALSE.
   *
   * @return array
   *   The uids that match the keys (no index).
   */
  public function whereMailFields2Uid(array $keys, $field = 'mail', $operator = 'LIKE', $insensitive_search = FALSE) {

    $db_funct = FALSE;
    if (empty($keys) || (!is_array($keys)) || (!in_array($field, ['mail', 'init']) && (!$db_funct = $this->dbFunctionsClause($field)))) {
      return [];
    }

    // Prepare the regex patterns corresponding to the key(s).
    $patterns = [];
    // Look for unescaped wildcards.
    // Find active '%' mysql wildcard, replaced by php regex '.*'.
    $pattern_percent = '[^\\\\]%|^%';
    // Find active '_' mysql wildcard, replaced by php regex '.'.
    $pattern_underscore = '[^\\\\]_|^_';
    $pattern_wildcards = "$pattern_percent|$pattern_underscore";
    $like = in_array($operator, ['LIKE', 'NOT LIKE']);
    foreach ($keys as $key) {
      // Prepare the key.
      $php_value = ($like && preg_match("!^%!", $key) === 1) ? '#' : '#^';
      if ($like) {

        $split_value = preg_split("!($pattern_wildcards)!", $key, -1, PREG_SPLIT_DELIM_CAPTURE | PREG_SPLIT_NO_EMPTY);
        $index_max = count($split_value) - 1;
        foreach ($split_value as $index => $string) {
          if (mb_strlen($string) <= 2) {
            $substring = $string;
            $replace_by = '';
            if (preg_match("!([^\\\\])%|()^%!", $string, $previous_char) === 1) {
              $replace_by = ($index > 0 && $index < $index_max) ? '.*' : '';
              $substring = $previous_char[1];
            }
            elseif (preg_match("!([^\\\\])_|()^_!", $string, $previous_char) === 1) {
              $replace_by = '.';
              $substring = $previous_char[1];
            }
            // db_like() add addcslashes() function.
            $php_value .= quotemeta(stripcslashes($substring)) . $replace_by;
          }
          else {
            $php_value .= quotemeta(stripcslashes($string));
          }
        }
      }
      else {
        $php_value .= quotemeta($key);
      }
      if ($insensitive_search) {
        $php_value = mb_strtolower($php_value);
      }

      $php_value .= ($like && preg_match("!%$!", $key) === 1) ? '#' : '$#';
      $patterns[] = $php_value;
    }

    // Decrypt all users.
    $users = _dbee_all_users_uncrypted();

    $result = [];
    foreach ($users as $uid => $datas) {
      if (!$db_funct) {
        $uncrypted_field = $datas[$field];
      }
      else {
        $uncrypted_field = $this->dbFunctionsValue($db_funct, $datas);
      }

      if ($insensitive_search) {
        $uncrypted_field = mb_strtolower($uncrypted_field);
      }
      // Search the key(s).
      foreach ($patterns as $pattern) {
        // Like or Not like never returns for null values.
        if (!empty($uncrypted_field) && preg_match($pattern, $uncrypted_field)) {
          // Avoid duplicate entry thanks to the index.
          $result[$uid] = $uid;
        }
      }
    }
    return array_values($result);
  }

  /**
   * Provides a list of possible mysql aliases for a field in a table.
   *
   * Consider alias values for field and table, handling if the field alias
   * includes the table name (case: $field_alias = "users.mail").
   *
   * @param string $field
   *   The exact field name in the database (example: "mail").
   * @param string $table
   *   The table name in the database (example: "drupal_user").
   *   Optional. Defaults to empty.
   * @param string $field_alias
   *   The alias field name in the query (example: "user_mail" or
   *   "user.mail"). Optional. Defaults to empty.
   * @param string $table_alias
   *   The alias table name in the query (example: "user_table").
   *   Optional. Defaults to empty.
   *
   * @return array
   *   All possible values that match the field according to the query
   *   (example: "mail", "user_mail", "user.mail", "user_table.mail",
   *   "user.user_mail", "user_table.user_mail").
   */
  public function buildAlias($field, $table = '', $field_alias = '', $table_alias = '') {
    // If there is no field, there is no alias.
    if (empty($field)) {
      return [];
    }
    // Handle the field.
    $alias = [$field];

    // Handle the field alias.
    if (empty($field_alias)) {
      $field_alias = $field;
    }
    $alias[] = $field_alias;

    // If a table has been provided.
    if (!empty($table)) {
      $alias[] = "{$table}.{$field}";

      // Handle the table alias.
      if (empty($table_alias)) {
        $table_alias = $table;
      }
      $alias[] = "{$table_alias}.{$field}";

      // Handle the field alias with the table.
      if (strpos($field_alias, '.') === FALSE) {
        // $field_alias does not contain the table.
        $alias[] = "{$table}.{$field_alias}";
        $alias[] = "{$table_alias}.{$field_alias}";
      }
    }

    // Format the result.
    $alias = array_unique($alias);
    $alias = array_values($alias);
    return $alias;
  }

  /**
   * Manages database functions in a conditional clause query.
   *
   * Special clauses can involve mysql or other database functions, especially
   * those from the admin/people page.
   *
   * @param string $clause
   *   A clause that may contain a database function, such as: "CONCAT_WS(' ',
   *   users_field_data.name, ' ', users_field_data.mail) LIKE :views_combine".
   *
   * @return array|bool
   *   Keys are: 'db_funct' (the database function name, example:
   *   'CONCACT_WS'), 'fields' (an array of field names) and 'op' (a string
   *   corresponding to the operator). Or FALSE if the clause does not contain a
   *   database function.
   */
  public function dbFunctionsClause($clause) {
    if (is_string($clause)) {
      // More DB function than CONCAT_WS can be managed. Their code will be
      // here. CONCAT_WS exists for both mysql and postgres.
      $db_functions = ['CONCAT_WS'];
      foreach ($db_functions as $db_funct) {
        $fields = $field_names = [];
        $op = FALSE;
        if (substr($clause, 0, strlen($db_funct)) == $db_funct) {
          // Search "CONCAT_WS(' ',ufd.name,' ', ufd.mail) LIKE :views_combine".
          $pos1 = strpos($clause, '(');
          $pos2 = strrpos($clause, ')');
          $in_brackets = substr($clause, ($pos1 + 1), ($pos2 - $pos1 - 1));
          $after_brackets = substr($clause, ($pos2 + 1));
          if (!empty($after_brackets)) {
            $op = trim($after_brackets);
            $op = trim(mb_strtoupper(substr($op, 0, strpos($op, ' '))));
          }
          if (!empty($in_brackets)) {
            $in_brackets_array = explode(',', $in_brackets);
            foreach ($in_brackets_array as $field) {
              // Do not delete space char.
              $field = trim($field);
              $field = (!in_array($field, [' ', "' '", '" "'])) ? $field : " ";
              $fields[] = $field;
              $dot_pos = strrpos($field, '.');
              $field_names[] = ($dot_pos === FALSE) ? $field : substr($field, ($dot_pos + 1));
            }
          }
        }
        if (!empty($fields) && !empty($op)) {
          return [
            'db_funct' => $db_funct,
            'fields' => $fields,
            'field_names' => $field_names,
            'op' => $op,
          ];
        }
      }
    }
    return FALSE;
  }

  /**
   * Converts database functions into values for php.
   *
   * @param array $db_funct
   *   The array returned by the dbFunctionsClause() function.
   * @param array $user
   *   An array of user information. Keys are mail, init, and name.
   *
   * @return string
   *   The value corresponding to the one that should be returned by the
   *   database query.
   *
   * @see dbFunctionsClause()
   */
  public function dbFunctionsValue(array $db_funct, array $user) {
    $funct = (isset($db_funct['db_funct'])) ? $db_funct['db_funct'] : NULL;
    switch ($funct) {
      case 'CONCAT_WS':
        // Note: more db_funct could be managed.
        $separator = array_shift($db_funct['field_names']);
        $values = [];
        foreach ($db_funct['field_names'] as $field) {
          $value = (in_array($field, ['mail', 'init', 'name'])) ? $user[$field] : $field;
          if ($value === NULL) {
            continue;
          }
          $values[] = $value;
        }
        $return = implode($separator, $values);
        break;

      default:
        $return = '';
    }
    return $return;
  }

}
