<?php

namespace Drupal\dbee\Plugin\Validation\Constraint;

use Drupal\Core\Validation\Plugin\Validation\Constraint\UniqueFieldValueValidator;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\Exception\UnexpectedTypeException;

/**
 * {@inheritdoc}
 */
class UserMailUniqueDbeeValidator extends UniqueFieldValueValidator {

  /**
   * {@inheritdoc}
   */
  public function validate($items, Constraint $constraint): void {

    if (!$constraint instanceof UserMailUniqueDbee) {
      throw new UnexpectedTypeException($constraint, UserMailUniqueDbee::class);
    }

    if (!$items->first()) {
      return;
    }

    /** @var \Drupal\Core\Entity\EntityInterface $entity */
    $entity = $items->getEntity();
    $entity_type = $entity->getEntityType();
    $entity_type_id = $entity_type->id();
    $entity_label = $entity->getEntityType()->getSingularLabel();

    $field_name = $items->getFieldDefinition()->getName();
    $field_label = $items->getFieldDefinition()->getLabel();
    $field_storage_definitions = $this->entityFieldManager->getFieldStorageDefinitions($entity_type_id);
    $property_name = $field_storage_definitions[$field_name]->getMainPropertyName();

    $id_key = $entity_type->getKey('id');
    $is_multiple = $field_storage_definitions[$field_name]->isMultiple();
    $is_new = $entity->isNew();
    $item_values = array_column($items->getValue(), $property_name);
    // Here the dbee Alteration
    // $item_values = [$entity->getEmail()];
    $item_values_decrypted = [];
    foreach ($item_values as $index => $mail) {
      $decrypted = dbee_decrypt($mail);
      $item_values_decrypted[$index] = $decrypted;
    }

    // Decrypt all users. Refresh the cache.
    $users = _dbee_all_users_uncrypted(TRUE);

    $target_uid = $entity->{$id_key}->value;
    foreach ($users as $uid => $datas) {
      if ($is_new || $uid != $target_uid) {
        $mail = $datas[$field_name];
        if ($mail) {
          foreach ($item_values_decrypted as $delta => $target_mail) {
            $original = $target_mail;
            if (!$constraint->caseSensitive) {
              $mail = mb_strtolower($mail);
              $target_mail = mb_strtolower($target_mail);
            }
            $duplicate = (!is_null($mail) && $mail === $target_mail);
            if ($duplicate) {
              $violation = $this->context
                ->buildViolation($constraint->message)
                ->setParameter('@entity_type', $entity_label)
                ->setParameter('@field_name', $field_label)
                ->setParameter('%value', $original);
              if ($is_multiple) {
                $violation->atPath((string) $delta);
              }
              $violation->addViolation();
            }
          }
        }
      }
    }

    // Check if items are duplicated within this entity.
    if ($is_multiple && count($item_values_decrypted) > 1) {
      $existings = [];
      foreach ($item_values_decrypted as $delta => $target_mail) {
        $original = $target_mail;
        if (!$constraint->caseSensitive) {
          $target_mail = mb_strtolower($target_mail);
        }
        if (in_array($target_mail, $existings)) {
          $this->context
            ->buildViolation($constraint->message)
            ->setParameter('@entity_type', $entity_label)
            ->setParameter('@field_name', $field_label)
            ->setParameter('%value', $original)
            ->atPath((string) $delta)
            ->addViolation();
        }
        // At the end, so ignore first.
        $existings[] = $target_mail;
      }
    }
  }

}
