/**
 * @file
 * JavaScript for real-time server clock update and element repositioning on the dblog page.
 */

(function ($, Drupal) {

  'use strict';

  /**
   * Updates the clock in real-time and repositions the clock element outside the form.
   */
  Drupal.behaviors.dblogTimeUpdater = {
    attach: function (context, settings) {
      // Find the time element container and ensure the script runs only once per page load.
      var $timeContainer = $('#dblog-time-reference', context).once('dblog-time-mover');

      if ($timeContainer.length) {
        // --- LOGIC TO MOVE ELEMENT OUTSIDE THE FORM ---

        // Find the containing Views exposed form. Views forms start with 'views-exposed-form-watchdog-'.
        var $form = $timeContainer.closest('form[id^="views-exposed-form-watchdog-"]');

        if ($form.length) {
          // Move the entire time container element (#dblog-time-reference)
          // to be immediately before the main <form> tag.
          $timeContainer.insertBefore($form);

          // Remove the once ID from the container to prevent re-running the move logic
          // if the form is re-AJAXed, as the container is no longer inside the form.
          $timeContainer.removeOnce('dblog-time-mover');
        }

        // --- REAL-TIME CLOCK UPDATE LOGIC ---

        var $timeElement = $('#dblog-time-value', $timeContainer);

        if ($timeElement.length) {
            // Get the server's timestamp on page load (in seconds). Convert to milliseconds.
            var serverLoadTimestamp = parseInt($timeElement.data('timestamp'), 10) * 1000;

            // Function to format a timestamp into a readable string (YYYY-MM-DD HH:MM:SS T).
            function formatTime(timestamp) {
              var date = new Date(timestamp);
              var year = date.getFullYear();
              var month = ('0' + (date.getMonth() + 1)).slice(-2);
              var day = ('0' + date.getDate()).slice(-2);
              var hours = ('0' + date.getHours()).slice(-2);
              var minutes = ('0' + date.getMinutes()).slice(-2);
              var seconds = ('0' + date.getSeconds()).slice(-2);

              // Attempt to get client timezone abbreviation.
              var timezone = date.toLocaleTimeString('en-us',{timeZoneName:'short'}).split(' ')[2] || 'Client';

              return year + '-' + month + '-' + day + ' ' + hours + ':' + minutes + ':' + seconds + ' ' + timezone;
            }

            // Calculate the initial offset between client load time and server load time.
            var clientLoadTimestamp = new Date().getTime();
            var offset = clientLoadTimestamp - serverLoadTimestamp;

            // Main update function.
            function updateTime() {
              // Estimated server time = Current client time - offset.
              var estimatedServerTime = new Date().getTime() - offset;

              $timeElement.text(formatTime(estimatedServerTime));
            }

            // Run immediately and then set the interval for 1 second.
            updateTime();
            // Clear any existing interval to prevent duplicates.
            clearInterval($timeElement.data('timerId'));
            var timerId = setInterval(updateTime, 1000);
            $timeElement.data('timerId', timerId); // Store ID for cleanup.
        }
      }
    }
  };

})(jQuery, Drupal);
