<?php

namespace Drupal\dboptimize\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 *
 */
class DbOptimizeForm extends FormBase {

  protected $database;

  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   *
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database')
    );
  }

  /**
   *
   */
  public function getFormId() {
    return 'dboptimize_form';
  }

  /**
   *
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $header = [
      'select' => ['data' => 'Select', 'class' => ['checkbox']],
      'name' => ['data' => 'Table name'],
      'rows' => ['data' => 'Rows'],
      'size' => ['data' => 'Size (MB)'],
      'free' => ['data' => 'Free space (MB)'],
    ];

    $tables = $this->database->query('SHOW TABLE STATUS')->fetchAll();

    $search = $form_state->getValue('search', '');

    $form['search'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search'),
      '#default_value' => $search,
      '#attributes' => [
        'placeholder' => $this->t('Table name'),
        'class' => ['dboptimize-filter'],
      ],
    ];

    $form['button_actions'] = [
      'select_all' => [
        '#type' => 'button',
        '#value' => $this->t('Select All'),
        '#attributes' => ['class' => ['dboptimize-select-all']],
      ],
      'toggle' => [
        '#type' => 'button',
        '#value' => $this->t('Toggle'),
        '#attributes' => ['class' => ['dboptimize-toggle']],
      ],
      'clear' => [
        '#type' => 'button',
        '#value' => $this->t('Clear All'),
        '#attributes' => ['class' => ['dboptimize-clear']],
      ],
    ];

    $form['tables'] = [
      '#type' => 'tableselect',
      '#header' => $header,
      '#options' => [],
      '#multiple' => TRUE,
      '#attributes' => ['class' => ['dboptimize-table']],
    ];

    foreach ($tables as $table) {
      if (!is_object($table) || (!empty($search) && stripos($table->Name, $search) === FALSE)) {
        continue;
      }

      $size_mb = round(($table->Data_length + $table->Index_length) / 1048576, 2);
      $free_mb = round($table->Data_free / 1048576, 2);

      $form['tables']['#options'][$table->Name] = [
        'select' => '',
        'name' => $table->Name,
        'rows' => $table->Rows,
        'size' => $size_mb . ' MB',
        'free' => $free_mb . ' MB',
      ];
    }

    $form['actions_submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Optimize selected tables'),
    ];

    $form['#attached']['library'][] = 'dboptimize/dboptimize';

    return $form;
  }

  /**
   *
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $selected_tables = array_filter($form_state->getValue('tables'));

    \Drupal::request()->getSession()->set('selected_tables', array_keys($selected_tables));
    $form_state->setRedirect('dboptimize.confirm');
  }

}
