<?php

namespace Drupal\dboptimize\Form;

use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 *
 */
class DbOptimizeConfirmForm extends ConfirmFormBase {

  /**
   *
   */
  public function getFormId() {
    return 'dboptimize_confirm_form';
  }

  /**
   *
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to optimize the selected tables?');
  }

  /**
   *
   */
  public function getCancelUrl() {
    return Url::fromRoute('dboptimize.optimize_form');
  }

  /**
   *
   */
  public function getConfirmText() {
    return $this->t('Confirm');
  }

  /**
   *
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $tables = \Drupal::request()->getSession()->get('selected_tables', []);

    $form['tables'] = [
      '#markup' => '<p>' . implode('<br>', array_map('htmlspecialchars', $tables)) . '</p>',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   *
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $tables = \Drupal::request()->getSession()->get('selected_tables', []);

    $batch = (new BatchBuilder())
      ->setTitle($this->t('Optimizing database tables'))
      ->setFinishCallback([static::class, 'optimizeFinished']);

    foreach ($tables as $table) {
      $batch->addOperation([static::class, 'optimizeTable'], [$table]);
    }

    batch_set($batch->toArray());
  }

  /**
   *
   */
  public static function optimizeTable($table, array &$context) {
    $size_before = \Drupal::database()->query("SHOW TABLE STATUS LIKE '{$table}'")->fetchObject();
    $free_before = $size_before->Data_free;

    \Drupal::database()->query("OPTIMIZE TABLE `{$table}`");

    $size_after = \Drupal::database()->query("SHOW TABLE STATUS LIKE '{$table}'")->fetchObject();
    $free_after = $size_after->Data_free;

    $freed_mb = round(($free_before - $free_after) / 1048576, 2);

    $context['results']['optimized'][] = ['table' => $table, 'freed' => $freed_mb];
    $context['message'] = t('Optimized @table: freed @freed MB', ['@table' => $table, '@freed' => $freed_mb]);
  }

  /**
   *
   */
  public static function optimizeFinished($success, $results, $operations) {
    $total_freed = 0;
    foreach ($results['optimized'] as $data) {
      $total_freed += $data['freed'];
    }

    \Drupal::messenger()->addMessage(t('Database optimization completed. Total space freed: @total MB.', ['@total' => round($total_freed, 2)]));

    $response = new RedirectResponse(Url::fromRoute('dboptimize.optimize_form')->toString());
    $response->send();
  }

}
