<?php

namespace Drupal\dboptimize\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a form for selecting and optimizing database tables.
 */
class DbOptimizeForm extends FormBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a new \Drupal\dboptimize\Form\DbOptimizeForm object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   */
  public function __construct(Connection $database, RequestStack $request_stack) {
    $this->database = $database;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('request_stack')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dboptimize_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $header = [
      'select' => ['data' => 'Select', 'class' => ['checkbox']],
      'name' => ['data' => 'Table name'],
      'rows' => ['data' => 'Rows'],
      'size' => ['data' => 'Size (MB)'],
      'free' => ['data' => 'Free space (MB)'],
    ];

    $tables = $this->database->query('SHOW TABLE STATUS')->fetchAll();

    $search = $form_state->getValue('search', '');

    $form['form'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Database Tables'),
      '#description' => $this->t('Select the tables you want to optimize. You can filter the list by typing in the search box.'),
    ];
    $form['form']['search'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search'),
      '#default_value' => $search,
      '#attributes' => [
        'placeholder' => $this->t('Table name'),
        'class' => ['dboptimize-filter'],
      ],
    ];

    $form['form']['button_actions'] = [
      'select_all' => [
        '#type' => 'button',
        '#value' => $this->t('Select All'),
        '#attributes' => ['class' => ['dboptimize-select-all']],
      ],
      'toggle' => [
        '#type' => 'button',
        '#value' => $this->t('Toggle'),
        '#attributes' => ['class' => ['dboptimize-toggle']],
      ],
      'clear' => [
        '#type' => 'button',
        '#value' => $this->t('Clear All'),
        '#attributes' => ['class' => ['dboptimize-clear']],
      ],
    ];

    $form['tables'] = [
      '#type' => 'tableselect',
      '#header' => $header,
      '#options' => [],
      '#multiple' => TRUE,
      '#attributes' => ['class' => ['dboptimize-table']],
    ];

    foreach ($tables as $table) {
      if (!is_object($table) || (!empty($search) && stripos($table->Name, $search) === FALSE)) {
        continue;
      }

      $size_mb = round(($table->Data_length + $table->Index_length) / 1048576, 2);
      $free_mb = round($table->Data_free / 1048576, 2);

      $form['tables']['#options'][$table->Name] = [
        'select' => '',
        'name' => $table->Name,
        'rows' => $table->Rows,
        'size' => $size_mb . ' MB',
        'free' => $free_mb . ' MB',
      ];
    }

    $form['actions_submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Optimize selected tables'),
    ];

    $form['#attached']['library'][] = 'dboptimize/dboptimize';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $selected_tables = array_filter($form_state->getValue('tables'));

    // Use dependency injection for session.
    $this->requestStack->getCurrentRequest()->getSession()->set('selected_tables', array_keys($selected_tables));
    $form_state->setRedirect('dboptimize.confirm');
  }

}
