<?php

namespace Drupal\dboptimize;

/**
 * Run database ANALYZE on specified tables.
 */
class DbAnalyzeCron {

  /**
   * Run database ANALYZE on specified tables.
   */
  public function __invoke($drush_command_tables = '') {
    $config = \Drupal::service('config.factory')->getEditable('dboptimize.settings');
    $tables = $config->get('cron_tables_analyze') ?? [];
    $frequency_seconds = (int) $config->get('cron_frequency_analyze') ?: 86400;
    $last_run = $config->get('last_run_analyze') ?: 0;
    $now = time();
    $analyzed = [];
    $errors = [];

    if (!$drush_command_tables) {
      if (($now - $last_run) < $frequency_seconds) {
        // Not time yet to run.
        return;
      }
    }

    if (empty($tables)) {
      $result = \Drupal::database()->query('SHOW TABLES');
      $tables = array_column($result->fetchAll(\PDO::FETCH_NUM), 0);
    }

    if ($drush_command_tables) {
      $tables = explode(',', $drush_command_tables);
    }

    foreach ($tables as $table) {
      if ($this->tableExists($table)) {
        try {
          \Drupal::database()->query("ANALYZE TABLE `{$table}`");
          $analyzed[] = $table;
        }
        catch (\Exception $e) {
          $errors[] = "$table: {$e->getMessage()}";
        }
      }
    }

    if (!empty($analyzed)) {
      \Drupal::logger('dboptimize')->info('Analyzed tables: @tables.', [
        '@tables' => implode(', ', $analyzed),
      ]);
    }

    if (!empty($errors)) {
      \Drupal::logger('dboptimize')->error('Errors during analyze: @errors', [
        '@errors' => implode('; ', $errors),
      ]);
    }

    $config->set('last_run_analyze', $now)->save();
  }

  /**
   * Check if a table exists in the database.
   */
  protected function tableExists($table_name) {
    $result = \Drupal::database()->query("SHOW TABLES LIKE :name", [':name' => $table_name])->fetchField();
    return (bool) $result;
  }

}
