<?php

namespace Drupal\dboptimize;

/**
 * Run database CHECK on specified tables.
 */
class DbCheckCron {

  /**
   * Run database CHECK on specified tables.
   */
  public function __invoke($drush_command_tables = '') {
    $config = \Drupal::service('config.factory')->getEditable('dboptimize.settings');
    $tables = $config->get('cron_tables_check') ?? [];
    $frequency_seconds = (int) $config->get('cron_frequency_check') ?: 604800;
    $last_run = $config->get('last_run_check') ?: 0;
    $now = time();
    $checked = [];
    $errors = [];

    if (!$drush_command_tables) {
      if (($now - $last_run) < $frequency_seconds) {
        // Not time yet to run.
        return;
      }
    }

    if (empty($tables)) {
      $result = \Drupal::database()->query('SHOW TABLES');
      $tables = array_column($result->fetchAll(\PDO::FETCH_NUM), 0);
    }

    if ($drush_command_tables) {
      $tables = explode(',', $drush_command_tables);
    }

    foreach ($tables as $table) {
      if ($this->tableExists($table)) {
        try {
          \Drupal::database()->query("CHECK TABLE `{$table}`");
          $checked[] = $table;
        }
        catch (\Exception $e) {
          $errors[] = "$table: {$e->getMessage()}";
        }
      }
    }

    if (!empty($checked)) {
      \Drupal::logger('dboptimize')->info('Checked tables: @tables.', [
        '@tables' => implode(', ', $checked),
      ]);
    }

    if (!empty($errors)) {
      \Drupal::logger('dboptimize')->error('Errors during check: @errors', [
        '@errors' => implode('; ', $errors),
      ]);
    }

    $config->set('last_run_check', $now)->save();
  }

  /**
   * Check if a table exists in the database.
   */
  protected function tableExists($table_name) {
    $result = \Drupal::database()->query("SHOW TABLES LIKE :name", [':name' => $table_name])->fetchField();
    return (bool) $result;
  }

}
