<?php

namespace Drupal\dboptimize\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;

/**
 * Provides a form for backing up the database.
 */
class DbOptimizeDbBackupForm extends FormBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dboptimize_backup_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $form['fieldset'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Database Backup'),
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
    ];

    $form['fieldset']['description0'] = [
      '#markup' => $this->t('For very large databases, this backup may fail due to PHP memory or execution time limits, or server restrictions. If your database is very large, prefer using command-line tools or ask your sysadmin for a direct backup. The PHP process may timeout or run out of memory during the dump or file transfer.'),
    ];

    $form['fieldset']['gzip'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Compress backup with gzip'),
      '#default_value' => TRUE,
    ];

    $form['fieldset']['backup'] = [
      '#type' => 'submit',
      '#value' => $this->t('Download Database Backup'),
    ];
    $form['fieldset']['description1'] = [
      '#markup' => $this->t('Click the button to download a full database backup.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $database = $this->database->getConnectionOptions();
    $gzip = $form_state->getValue('gzip');

    $db_name = $database['database'];
    $date_str = date('Y-m-d__H-i-s');
    $tmp_file = sys_get_temp_dir() . '/backup_' . $db_name . '_' . $date_str . '.sql';
    $file_to_send = $tmp_file;

    // Detect mysqldump path.
    $mysqldump_path = trim(shell_exec('which mysqldump'));
    if (empty($mysqldump_path) || !is_executable($mysqldump_path)) {
      $this->messenger()->addError($this->t('mysqldump not found or not executable on server.'));
      return;
    }

    // Ensure sys_get_temp_dir() is writable.
    if (!is_writable(dirname($tmp_file))) {
      $this->messenger()->addError($this->t('Temporary directory is not writable.'));
      return;
    }

    // MySQL dump command.
    // Try disabling SSL with --skip-ssl for older mysqldump versions.
    $command = sprintf(
      '%s --user=%s --password=%s --host=%s --port=%s --single-transaction --skip-ssl %s > %s 2>&1',
      escapeshellcmd($mysqldump_path),
      escapeshellarg($database['username']),
      escapeshellarg($database['password']),
      escapeshellarg($database['host']),
      isset($database['port']) ? escapeshellarg($database['port']) : '3306',
      escapeshellarg($database['database']),
      escapeshellarg($tmp_file)
    );

    // Check if the database host is accessible from the web server
    // Optionally, you can test with: ping or nc (netcat) before
    // running mysqldump.
    exec($command, $output, $return_var);

    // Check for errors in output.
    $error_output = file_exists($tmp_file) ? file_get_contents($tmp_file) : '';

    if ($return_var !== 0 || strpos($error_output, 'mysqldump:') !== FALSE || strpos($error_output, 'No such file or directory') !== FALSE) {
      $this->messenger()->addError($this->t('Database backup failed. Command: @cmd', ['@cmd' => $command]));
      if (file_exists($tmp_file)) {
        unlink($tmp_file);
      }
      return;
    }

    // Optionally gzip.
    if ($gzip && file_exists($tmp_file)) {
      $gz_file = $tmp_file . '.gz';
      $gz = gzopen($gz_file, 'wb9');
      gzwrite($gz, file_get_contents($tmp_file));
      gzclose($gz);
      unlink($tmp_file);
      $file_to_send = $gz_file;
    }

    if (file_exists($file_to_send)) {
      $response = new BinaryFileResponse($file_to_send);
      $response->setContentDisposition(
        ResponseHeaderBag::DISPOSITION_ATTACHMENT,
        basename($file_to_send)
      );
      $response->deleteFileAfterSend(TRUE);
      $response->send();
      exit;
    }
    else {
      $this->messenger()->addError($this->t('Database backup failed. File not created.'));
    }
  }

}
