<?php

namespace Drupal\dboptimize\Form;

use Drupal\Core\Database\Database;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure database optimization settings.
 */
class DbOptimizeSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dboptimize_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['dboptimize.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('dboptimize.settings');

    // Get the list of database tables.
    $tables = Database::getConnection()->query('SHOW TABLES')->fetchCol();
    $options = array_combine($tables, $tables);

    // Optimize fieldset.
    $form['optimize'] = [
      '#type' => 'details',
      '#title' => $this->t('Optimize Settings'),
    ];
    $form['optimize']['message'] = [
      '#markup' => '<p>' . $this->t('Configure cron for table optimization.') . '</p>',
    ];
    $form['optimize']['cron_frequency'] = [
      '#type' => 'select',
      '#title' => $this->t('Cron frequency'),
      '#description' => $this->t('How often should the cron optimize the tables?'),
      '#options' => [
        3600 => $this->t('Hourly'),
        86400 => $this->t('Daily'),
        604800 => $this->t('Weekly'),
        2629746 => $this->t('Monthly'),
        7889238 => $this->t('Trimester (3 months)'),
        15778476 => $this->t('Semester (6 months)'),
        31556952 => $this->t('Yearly'),
      ],
      '#default_value' => $config->get('cron_frequency') ?? 86400,
    ];
    $form['optimize']['cron_tables'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Tables to optimize during cron'),
      '#description' => $this->t('Select the tables that should be optimized. If none are selected, all database tables will be optimized.'),
      '#options' => $options,
      '#default_value' => $config->get('cron_tables') ?? [],
    ];
    $form['optimize']['message_note'] = [
      '#markup' => '<p><strong>Note:</strong> If you specify one or more table names above, the cron will optimize only those tables. If you leave this field empty, all database tables will be optimized during cron execution.</p>',
    ];

    // Analyze fieldset.
    $form['analyze'] = [
      '#type' => 'details',
      '#title' => $this->t('Analyze Settings'),
    ];
    $form['analyze']['cron_frequency_analyze'] = [
      '#type' => 'select',
      '#title' => $this->t('Cron frequency'),
      '#description' => $this->t('How often should the cron analyze the tables?'),
      '#options' => [
        3600 => $this->t('Hourly'),
        86400 => $this->t('Daily'),
        604800 => $this->t('Weekly'),
        2629746 => $this->t('Monthly'),
        7889238 => $this->t('Trimester (3 months)'),
        15778476 => $this->t('Semester (6 months)'),
        31556952 => $this->t('Yearly'),
      ],
      '#default_value' => $config->get('cron_frequency_analyze') ?? 86400,
    ];
    $form['analyze']['cron_tables_analyze'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Tables to analyze during cron'),
      '#description' => $this->t('Select the tables that should be analyzed. If none are selected, all database tables will be analyzed during cron execution.'),
      '#options' => $options,
      '#default_value' => $config->get('cron_tables_analyze') ?? [],
    ];

    // Repair fieldset.
    $form['repair'] = [
      '#type' => 'details',
      '#title' => $this->t('Repair Settings'),
    ];
    $form['repair']['cron_frequency_repair'] = [
      '#type' => 'select',
      '#title' => $this->t('Cron frequency'),
      '#description' => $this->t('How often should the cron repair the tables?'),
      '#options' => [
        3600 => $this->t('Hourly'),
        86400 => $this->t('Daily'),
        604800 => $this->t('Weekly'),
        2629746 => $this->t('Monthly'),
        7889238 => $this->t('Trimester (3 months)'),
        15778476 => $this->t('Semester (6 months)'),
        31556952 => $this->t('Yearly'),
      ],
      '#default_value' => $config->get('cron_frequency_repair') ?? 604800,
    ];
    $form['repair']['cron_tables_repair'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Tables to repair during cron'),
      '#description' => $this->t('Select the tables that should be repaired. If none are selected, all database tables will be repaired during cron execution.'),
      '#options' => $options,
      '#default_value' => $config->get('cron_tables_repair') ?? [],
    ];

    // Check fieldset.
    $form['check'] = [
      '#type' => 'details',
      '#title' => $this->t('Check Settings'),
    ];
    $form['check']['cron_frequency_check'] = [
      '#type' => 'select',
      '#title' => $this->t('Cron frequency'),
      '#description' => $this->t('How often should the cron check the tables?'),
      '#options' => [
        3600 => $this->t('Hourly'),
        86400 => $this->t('Daily'),
        604800 => $this->t('Weekly'),
        2629746 => $this->t('Monthly'),
        7889238 => $this->t('Trimester (3 months)'),
        15778476 => $this->t('Semester (6 months)'),
        31556952 => $this->t('Yearly'),
      ],
      '#default_value' => $config->get('cron_frequency_check') ?? 604800,
    ];
    $form['check']['cron_tables_check'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Tables to check during cron'),
      '#description' => $this->t('Select the tables that should be checked. If none are selected, all database tables will be checked during cron execution.'),
      '#options' => $options,
      '#default_value' => $config->get('cron_tables_check') ?? [],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Optimize.
    $selected_tables = array_filter($form_state->getValue('cron_tables'));
    $cron_frequency = $form_state->getValue('cron_frequency');
    // Analyze.
    $selected_tables_analyze = array_filter($form_state->getValue('cron_tables_analyze'));
    $cron_frequency_analyze = $form_state->getValue('cron_frequency_analyze');
    // Repair.
    $selected_tables_repair = array_filter($form_state->getValue('cron_tables_repair'));
    $cron_frequency_repair = $form_state->getValue('cron_frequency_repair');
    // Check.
    $selected_tables_check = array_filter($form_state->getValue('cron_tables_check'));
    $cron_frequency_check = $form_state->getValue('cron_frequency_check');

    $this->config('dboptimize.settings')
      ->set('cron_tables', $selected_tables)
      ->set('cron_frequency', $cron_frequency)
      ->set('cron_tables_analyze', $selected_tables_analyze)
      ->set('cron_frequency_analyze', $cron_frequency_analyze)
      ->set('cron_tables_repair', $selected_tables_repair)
      ->set('cron_frequency_repair', $cron_frequency_repair)
      ->set('cron_tables_check', $selected_tables_check)
      ->set('cron_frequency_check', $cron_frequency_check)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
