<?php

namespace Drupal\dboptimize;

/**
 * Class DbOptimizeCron.
 *
 * Provide cron functionality for dboptimize module.
 */
class DbOptimizeCron {

  /**
   * Run database optimization on specified tables.
   */
  public function __invoke($drush_command_tables = '') {
    $config = \Drupal::service('config.factory')->getEditable('dboptimize.settings');
    $tables = $config->get('cron_tables') ?? [];
    $frequency_seconds = (int) $config->get('cron_frequency') ?: 86400;
    $last_run = $config->get('last_run') ?: 0;
    $now = time();
    $optimized = [];
    $errors = [];
    $freed_mb = [];

    if (!$drush_command_tables) {
      if (($now - $last_run) < $frequency_seconds) {
        // Not time yet to run.
        return;
      }
    }

    if (empty($tables)) {
      $result = \Drupal::database()->query('SHOW TABLES');
      $tables = array_column($result->fetchAll(\PDO::FETCH_NUM), 0);
    }

    if ($drush_command_tables) {
      $tables = explode(',', $drush_command_tables);
    }

    foreach ($tables as $table) {
      if ($this->tableExists($table)) {
        try {
          $size_before = \Drupal::database()->query("SHOW TABLE STATUS LIKE '{$table}'")->fetchObject();
          $free_before = $size_before->Data_free ?? 0;

          \Drupal::database()->query("OPTIMIZE TABLE `{$table}`");

          $optimized[] = $table;
          $size_after = \Drupal::database()->query("SHOW TABLE STATUS LIKE '{$table}'")->fetchObject();
          $free_after = $size_after->Data_free ?? 0;

          $freed_mb[] = round(($free_before - $free_after) / 1048576, 2);
        }
        catch (\Exception $e) {
          $errors[] = "$table: {$e->getMessage()}";
        }
      }
    }

    if ($freed_mb) {
      $freed_mb = array_sum($freed_mb);
    }

    if (!empty($optimized)) {
      \Drupal::logger('dboptimize')->info('Optimized tables: @tables. <br><b>Freed: @total</b>', [
        '@tables' => implode(', ', $optimized),
        '@total' => $freed_mb,
      ]);
    }

    if (!empty($errors)) {
      \Drupal::logger('dboptimize')->error('Errors during optimization: @errors', [
        '@errors' => implode('; ', $errors),
      ]);
    }

    $config->set('last_run', $now)->save();
  }

  /**
   * Check if a table exists in the database.
   */
  protected function tableExists($table_name) {
    $result = \Drupal::database()->query("SHOW TABLES LIKE :name", [':name' => $table_name])->fetchField();
    return (bool) $result;
  }

}
