<?php

namespace Drupal\dboptimize;

/**
 * Run database REPAIR on specified tables.
 */
class DbRepairCron {

  /**
   * Run database REPAIR on specified tables.
   */
  public function __invoke($drush_command_tables = '') {
    $config = \Drupal::service('config.factory')->getEditable('dboptimize.settings');
    $tables = $config->get('cron_tables_repair') ?? [];
    $frequency_seconds = (int) $config->get('cron_frequency_repair') ?: 604800;
    $last_run = $config->get('last_run_repair') ?: 0;
    $now = time();
    $repaired = [];
    $errors = [];

    if (!$drush_command_tables) {
      if (($now - $last_run) < $frequency_seconds) {
        // Not time yet to run.
        return;
      }
    }

    if (empty($tables)) {
      $result = \Drupal::database()->query('SHOW TABLES');
      $tables = array_column($result->fetchAll(\PDO::FETCH_NUM), 0);
    }

    if ($drush_command_tables) {
      $tables = explode(',', $drush_command_tables);
    }

    foreach ($tables as $table) {
      if ($this->tableExists($table)) {
        try {
          \Drupal::database()->query("REPAIR TABLE `{$table}`");
          $repaired[] = $table;
        }
        catch (\Exception $e) {
          $errors[] = "$table: {$e->getMessage()}";
        }
      }
    }

    if (!empty($repaired)) {
      \Drupal::logger('dboptimize')->info('Repaired tables: @tables.', [
        '@tables' => implode(', ', $repaired),
      ]);
    }

    if (!empty($errors)) {
      \Drupal::logger('dboptimize')->error('Errors during repair: @errors', [
        '@errors' => implode('; ', $errors),
      ]);
    }

    $config->set('last_run_repair', $now)->save();
  }

  /**
   * Check if a table exists in the database.
   */
  protected function tableExists($table_name) {
    $result = \Drupal::database()->query("SHOW TABLES LIKE :name", [':name' => $table_name])->fetchField();
    return (bool) $result;
  }

}
