<?php

namespace Drupal\dc_ajax_add_cart_popup\EventSubscriber;

use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Drupal\commerce_cart\Event\CartEntityAddEvent;
use Drupal\commerce_cart\Event\CartEvents;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\OpenModalDialogCommand;
use Drupal\Core\Url;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Event subscriber to display a popup when items are added to cart via AJAX.
 */
class AjaxAddToCartPopupSubscriber implements EventSubscriberInterface {

  /**
   * The entity that was added to the cart.
   *
   * @var \Drupal\commerce_product\Entity\ProductVariationInterface
   */
  protected $purchasedEntity;

  /**
   * Constructs a new AjaxAddToCartPopupSubscriber object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      KernelEvents::RESPONSE => 'onResponse',
      CartEvents::CART_ENTITY_ADD => 'onAddToCart',
    ];
  }

  /**
   * Adds the popup confirmation message on page.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event.
   */
  public function onResponse(ResponseEvent $event): void {
    $response = $event->getResponse();

    // We only care if this happened after an entity was added to the cart.
    if (!$this->purchasedEntity) {
      return;
    }

    // We only care about AJAX responses.
    if (!$response instanceof AjaxResponse) {
      return;
    }

    // Render the status message and the entity.
    $view_builder = $this->entityTypeManager->getViewBuilder('commerce_product_variation');
    $product_variation = $view_builder->view($this->purchasedEntity, 'dc_ajax_add_cart_popup');
    $content = [
      '#theme' => 'dc_ajax_add_cart_popup',
      '#product_variation' => $product_variation,
      '#product_variation_entity' => $this->purchasedEntity,
      '#cart_url' => Url::fromRoute('commerce_cart.page')->toString(),
    ];
    $title = '';
    $options = ['width' => '700'];
    $response->addCommand(new OpenModalDialogCommand($title, $content, $options));
    $event->setResponse($response);
  }

  /**
   * Initializes the purchased entity.
   *
   * @param \Drupal\commerce_cart\Event\CartEntityAddEvent $event
   *   The add to cart event.
   */
  public function onAddToCart(CartEntityAddEvent $event): void {
    $this->purchasedEntity = $event->getEntity();
  }

}
