# DCR Renderer Module

The DCR Renderer module provides a Twig extension and Single Directory Component (SDC) for fetching and rendering content from Drupal Content Repository URLs or local file paths. It includes multiple processing options extracted from the static_content_type module.

## Features

- **Twig Extension**: `dcr_get_contents()` function for fetching and processing remote/local content
- **SDC Component**: `dcr-render` component for embedding content with fallback options
- **Multiple Processing Types**: Raw, proxied, hardened, and full processing options
- **Asset Extraction**: Automatically handles CSS and JavaScript from fetched content
- **Security Controls**: Domain restrictions, file extension filtering, and content sanitization
- **Caching**: Configurable caching for improved performance
- **Error Handling**: Graceful fallbacks when content cannot be fetched

## Installation

1. Copy the module to your `modules/custom/dcr_renderer/` directory
2. Enable the module: `drush en dcr_renderer`
3. Configure settings at `/admin/config/content/dcr-renderer`

## Usage

### Twig Extension

Use the `dcr_get_contents()` function in any Twig template:

```twig
{# Basic usage - fetches and processes with default 'hardened' mode #}
{{ dcr_get_contents('https://example.com/repository/node/123') }}

{# With specific processing type #}
{{ dcr_get_contents('https://example.com/repository/node/123', 'proxied') }}

{# Local file #}
{{ dcr_get_contents('public://dcr-content/page.html', 'hardened') }}

{# Render array version for advanced control #}
{% set content = dcr_get_contents_render_array(url, 'hardened') %}
{{ content }}
```

### SDC Component

Use the `dcr-render` SDC component:

```twig
{# Basic SDC usage #}
{{ include('dcr_renderer:dcr-render', {
  url: 'https://example.com/repository/node/123',
  processing_type: 'hardened'
}) }}

{# With error fallback #}
{{ include('dcr_renderer:dcr-render', {
  url: 'https://example.com/repository/node/123',
  processing_type: 'hardened',
  error_fallback: '<p>Content temporarily unavailable.</p>'
}) }}

{# With slot fallback #}
{% embed 'dcr_renderer:dcr-render' with {
  url: 'https://example.com/repository/node/123'
} %}
  {% block fallback %}
    <div class="error-message">
      <p>Unable to load content. Please try again later.</p>
    </div>
  {% endblock %}
{% endembed %}
```

## Processing Types

### Raw
- Returns content exactly as fetched
- No URL processing or sanitization
- Fastest option but least secure

### Proxied
- Fixes relative URLs to point to the source domain
- Preserves complete HTML structure
- Good for content with working relative paths

### Hardened (Recommended)
- Extracts body content from HTML documents
- Fixes relative URLs to source domain
- Separates CSS and JavaScript for proper head injection
- Sanitizes HTML tags using allowlist
- Best security and compatibility

### Full
- Returns complete HTML document with URL fixes
- Useful for embedding complete pages in iframes
- Maintains original document structure

## URL Types Supported

- **Remote DCR URLs**: `https://example.com/repository/node/123`
- **Remote SDC URLs**: `https://example.com/sdc/theme/component?prop=value`
- **Local files**: `public://dcr-content/page.html`
- **Absolute paths**: `/var/www/content/page.html`
- **Relative paths**: `dcr-content/page.html` (relative to public files)

## Configuration

Configure the module at `/admin/config/content/dcr-renderer`:

### Processing Settings
- **Default Processing Type**: Choose the default processing mode
- **HTTP Timeout**: Timeout for remote requests (1-300 seconds)
- **User Agent**: Custom user agent for HTTP requests

### Security Settings
- **Allowed Domains**: Restrict remote fetching to specific domains
- **Allow Local Files**: Enable/disable local file access
- **File Extensions**: Restrict local files to specific extensions

### Caching Settings
- **Default Cache Duration**: How long to cache fetched content
- **Cache Errors**: Whether to cache failed requests

## Asset Handling

The hardened processing type automatically extracts and properly handles:

- **CSS Files**: `<link rel="stylesheet">` tags moved to document head
- **JavaScript Files**: `<script src="">` tags moved to document head  
- **Inline CSS**: `<style>` blocks moved to document head
- **Inline JavaScript**: `<script>` blocks moved to document head

This ensures proper rendering while maintaining Drupal's asset management.

## Security Considerations

### Hardened Mode (Recommended)
- Extracts only body content from HTML documents
- Sanitizes HTML using comprehensive tag allowlist
- Fixes relative URLs to prevent broken assets
- Separates CSS/JS for proper head injection

### Domain Restrictions
Configure allowed domains to prevent fetching from unauthorized sources:
```
example.com
*.mydomain.com
subdomain.example.org
```

### File Extension Filtering
Restrict local files to safe extensions:
```
html,htm,txt,xml
```

## Error Handling

The module provides graceful error handling:

- **Network errors**: Timeout or connection failures
- **HTTP errors**: 404, 500, etc. responses  
- **Local file errors**: Missing or unreadable files
- **Parsing errors**: Invalid HTML or processing failures

Error messages are wrapped in CSS classes for styling:
- `.dcr-error` - General error wrapper
- `.dcr-fallback-content` - Fallback content wrapper

## Performance

### Caching
- Fetched content is cached based on configuration
- Cache duration can be set per-request or globally
- Error responses can optionally be cached

### HTTP Client
- Uses Drupal's HTTP client with configurable timeout
- Follows redirects automatically
- Handles SSL certificates properly

## Development Workflow

1. **Content Creation**: Create content in external systems or local files
2. **URL Generation**: Generate DCR repository URLs or local paths
3. **Template Integration**: Use Twig extension or SDC in templates
4. **Processing Selection**: Choose appropriate processing type
5. **Error Handling**: Implement fallbacks for failed requests

## Troubleshooting

### Content Not Loading
- Check URL accessibility in browser
- Verify domain restrictions in configuration
- Check file permissions for local files
- Review error logs at `/admin/reports/dblog`

### Broken Assets (Images, CSS)
- Use 'proxied' or 'hardened' processing types
- Verify source content has proper relative paths
- Check if assets exist at source location

### Security Warnings
- Review allowed domains configuration
- Consider disabling local file access if not needed
- Restrict file extensions to safe types

## Module Structure

```
dcr_renderer/
├── dcr_renderer.info.yml           # Module definition
├── dcr_renderer.module             # Hook implementations and helper functions
├── dcr_renderer.services.yml       # Service definitions
├── dcr_renderer.routing.yml        # Admin routes
├── dcr_renderer.install            # Install/uninstall hooks
├── config/
│   ├── install/
│   │   └── dcr_renderer.settings.yml
│   └── schema/
│       └── dcr_renderer.schema.yml
├── src/
│   ├── Form/
│   │   └── DcrRendererConfigForm.php
│   └── TwigExtension/
│       └── DcrRendererTwigExtension.php
├── components/
│   └── dcr-render/
│       ├── dcr-render.component.yml
│       └── dcr-render.twig
└── templates/
    └── dcr-renderer-content.html.twig
```

## Requirements

- Drupal 11
- PHP cURL extension (recommended for HTTP requests)
- Write access to public files directory (for local content)

## License

GPL-2.0-or-later

## Support

For issues and feature requests, please use the project's issue queue or contact the maintainer.