<?php

namespace Drupal\dcr_renderer\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure DCR Renderer settings for this site.
 */
class DcrRendererConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'dcr_renderer_admin_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['dcr_renderer.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('dcr_renderer.settings');

    $form['description'] = [
      '#markup' => '<p>' . $this->t('Configure global settings for DCR Renderer module. This module allows fetching and rendering content from Drupal Content Repository URLs with various processing options.') . '</p>',
    ];

    $form['default_processing'] = [
      '#type' => 'select',
      '#title' => $this->t('Default Processing Type'),
      '#description' => $this->t('Default processing type when none is specified.'),
      '#options' => [
        'raw' => $this->t('Raw - No processing (fastest, least secure)'),
        'proxied' => $this->t('Proxied - Path fixing only (faster)'),
        'hardened' => $this->t('Hardened - Security sanitization + path proxying (safest)'),
        'full' => $this->t('Full - Complete document with path proxying'),
      ],
      '#default_value' => $config->get('default_processing') ?: 'hardened',
      '#required' => TRUE,
    ];

    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced Settings'),
      '#open' => FALSE,
    ];

    $form['advanced']['timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('HTTP Timeout'),
      '#description' => $this->t('Timeout in seconds for HTTP requests to remote URLs.'),
      '#default_value' => $config->get('timeout') ?: 30,
      '#min' => 1,
      '#max' => 300,
    ];

    $form['advanced']['user_agent'] = [
      '#type' => 'textfield',
      '#title' => $this->t('User Agent'),
      '#description' => $this->t('User agent string to send with HTTP requests.'),
      '#default_value' => $config->get('user_agent') ?: 'Drupal DCR Renderer Module',
      '#maxlength' => 255,
    ];

    $form['advanced']['allowed_domains'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Allowed Domains'),
      '#description' => $this->t('List of allowed domains for remote content fetching, one per line. Leave empty to allow all domains. Use * as wildcard (e.g., *.example.com).'),
      '#default_value' => $config->get('allowed_domains') ?: '',
      '#rows' => 5,
    ];

    $form['caching'] = [
      '#type' => 'details',
      '#title' => $this->t('Caching Settings'),
      '#open' => FALSE,
    ];

    $form['caching']['default_cache_duration'] = [
      '#type' => 'number',
      '#title' => $this->t('Default Cache Duration'),
      '#description' => $this->t('Default cache duration in seconds for fetched content.'),
      '#default_value' => $config->get('default_cache_duration') ?: 300,
      '#min' => 0,
      '#max' => 86400,
    ];

    $form['caching']['cache_errors'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Cache Errors'),
      '#description' => $this->t('Whether to cache error responses to avoid repeated failed requests.'),
      '#default_value' => $config->get('cache_errors') ?: FALSE,
    ];

    // Security settings.
    $form['security'] = [
      '#type' => 'details',
      '#title' => $this->t('Security Settings'),
      '#open' => FALSE,
    ];

    $form['security']['allow_local_files'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow Local File Access'),
      '#description' => $this->t('Allow fetching content from local file paths. Disable for security if only remote URLs should be allowed.'),
      '#default_value' => $config->get('allow_local_files') !== FALSE,
    ];

    $form['security']['restrict_file_extensions'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Allowed File Extensions'),
      '#description' => $this->t('Comma-separated list of allowed file extensions for local files (e.g., html,htm,txt). Leave empty to allow all.'),
      '#default_value' => $config->get('restrict_file_extensions') ?: 'html,htm',
    ];

    // Help section.
    $form['help'] = [
      '#type' => 'details',
      '#title' => $this->t('Usage Help'),
      '#open' => FALSE,
    ];

    $form['help']['info'] = [
      '#markup' => $this->getHelpText(),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('dcr_renderer.settings')
      ->set('default_processing', $form_state->getValue('default_processing'))
      ->set('timeout', $form_state->getValue('timeout'))
      ->set('user_agent', $form_state->getValue('user_agent'))
      ->set('allowed_domains', $form_state->getValue('allowed_domains'))
      ->set('default_cache_duration', $form_state->getValue('default_cache_duration'))
      ->set('cache_errors', $form_state->getValue('cache_errors'))
      ->set('allow_local_files', $form_state->getValue('allow_local_files'))
      ->set('restrict_file_extensions', $form_state->getValue('restrict_file_extensions'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Get help text for the module.
   */
  private function getHelpText() {
    return '
      <h3>Twig Extension Usage</h3>
      <p>Use the DCR Renderer Twig extension in any template:</p>
      <pre>
{# Basic usage #}
{{ dcr_get_contents("https://example.com/repository/node/123") }}

{# With processing type #}
{{ dcr_get_contents("https://example.com/repository/node/123", "hardened") }}

{# Local file #}
{{ dcr_get_contents("public://dcr-content/page.html", "proxied") }}

{# Render array version for advanced control #}
{% set content = dcr_get_contents_render_array(url, "hardened") %}
{{ content }}
      </pre>

      <h3>SDC Component Usage</h3>
      <p>Use the dcr-render SDC component:</p>
      <pre>
{# Basic SDC usage #}
{{ include("dcr_renderer:dcr-render", {
  url: "https://example.com/repository/node/123",
  processing_type: "hardened"
}) }}

{# With fallback content #}
{{ include("dcr_renderer:dcr-render", {
  url: "https://example.com/repository/node/123",
  processing_type: "hardened",
  error_fallback: "&lt;p&gt;Content temporarily unavailable.&lt;/p&gt;"
}) }}

{# With slot fallback #}
{% embed "dcr_renderer:dcr-render" with {
  url: "https://example.com/repository/node/123"
} %}
  {% block fallback %}
    &lt;div class="error-message"&gt;
      &lt;p&gt;Unable to load content. Please try again later.&lt;/p&gt;
    &lt;/div&gt;
  {% endblock %}
{% endembed %}
      </pre>

      <h3>Processing Types</h3>
      <ul>
        <li><strong>raw</strong> - Returns content exactly as fetched, no processing</li>
        <li><strong>proxied</strong> - Fixes relative URLs to point to the source domain</li>
        <li><strong>hardened</strong> - Extracts body content and fixes URLs, separates CSS/JS for head injection</li>
        <li><strong>full</strong> - Returns complete HTML document with URL fixes</li>
      </ul>

      <h3>URL Examples</h3>
      <ul>
        <li><strong>Remote DCR URLs:</strong> https://example.com/repository/node/123</li>
        <li><strong>Remote SDC URLs:</strong> https://example.com/sdc/theme/component-name?prop=value</li>
        <li><strong>Local files:</strong> public://dcr-content/page.html</li>
        <li><strong>Absolute paths:</strong> /var/www/content/page.html</li>
        <li><strong>Relative paths:</strong> dcr-content/page.html (relative to public files)</li>
      </ul>

      <h3>Security Considerations</h3>
      <p>The hardened processing type provides the best security by:</p>
      <ul>
        <li>Extracting only body content from HTML documents</li>
        <li>Sanitizing and whitelisting HTML tags</li>
        <li>Moving CSS and JavaScript to proper head attachment</li>
        <li>Fixing relative URLs to prevent broken assets</li>
      </ul>
    ';
  }

}