<?php

namespace Drupal\dcr_renderer\TwigExtension;

use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;

/**
 * Twig extension for DCR content fetching and processing.
 */
class DcrRendererTwigExtension extends AbstractExtension implements ContainerInjectionInterface {

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new DcrRendererTwigExtension.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(LoggerChannelFactoryInterface $logger_factory, ConfigFactoryInterface $config_factory) {
    $this->loggerFactory = $logger_factory;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('logger.factory'),
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFunctions() {
    return [
      new TwigFunction('dcr_get_contents', [$this, 'getDcrContents'], [
        'is_safe' => ['html'],
      ]),
      new TwigFunction('dcr_get_contents_render_array', [$this, 'getDcrContentsRenderArray']),
    ];
  }

  /**
   * Fetch and process content from a DCR URL or local path.
   *
   * @param string $url
   *   The URL or local path to fetch content from.
   * @param string $processing_type
   *   The processing type: 'raw', 'proxied', 'hardened', or 'full'.
   *
   * @return \Twig\Markup
   *   The processed content as safe markup.
   */
  public function getDcrContents($url, $processing_type = 'hardened') {
    $logger = $this->loggerFactory->get('dcr_renderer');
    $config = $this->configFactory->get('dcr_renderer.settings');

    $logger->info('DCR fetch request: @url with @type processing', [
      '@url' => $url,
      '@type' => $processing_type,
    ]);

    // Validate processing type.
    $valid_types = ['raw', 'proxied', 'hardened', 'full'];
    if (!in_array($processing_type, $valid_types)) {
      $processing_type = 'hardened';
      $logger->warning('Invalid processing type, defaulting to hardened');
    }

    // Fetch content.
    $content = $this->fetchContent($url);
    if ($content === FALSE) {
      $error_msg = 'Failed to fetch content from: ' . $url;
      $logger->error($error_msg);
      return new \Twig\Markup('<div class="dcr-error">Error: ' . htmlspecialchars($error_msg) . '</div>', 'UTF-8');
    }

    // Process content based on type.
    $processed_content = $this->processContent($content, $url, $processing_type);

    return new \Twig\Markup($processed_content, 'UTF-8');
  }

  /**
   * Fetch and process content returning a render array.
   *
   * @param string $url
   *   The URL or local path to fetch content from.
   * @param string $processing_type
   *   The processing type: 'raw', 'proxied', 'hardened', or 'full'.
   *
   * @return array
   *   A Drupal render array.
   */
  public function getDcrContentsRenderArray($url, $processing_type = 'hardened') {
    $logger = $this->loggerFactory->get('dcr_renderer');

    $logger->info('DCR render array request: @url with @type processing', [
      '@url' => $url,
      '@type' => $processing_type,
    ]);

    // Validate processing type.
    $valid_types = ['raw', 'proxied', 'hardened', 'full'];
    if (!in_array($processing_type, $valid_types)) {
      $processing_type = 'hardened';
      $logger->warning('Invalid processing type, defaulting to hardened');
    }

    // Fetch content.
    $content = $this->fetchContent($url);
    if ($content === FALSE) {
      $error_msg = 'Failed to fetch content from: ' . $url;
      $logger->error($error_msg);
      return [
        '#markup' => '<div class="dcr-error">Error: ' . htmlspecialchars($error_msg) . '</div>',
      ];
    }

    // Build render array based on processing type.
    $build = [];

    switch ($processing_type) {
      case 'raw':
        $build['#markup'] = $content;
        $build['#allowed_tags'] = dcr_renderer_get_allowed_tags();
        break;

      case 'proxied':
        $base_url = $this->getBaseUrl($url);
        $processed_content = dcr_renderer_process_proxied_content($content, $base_url);
        $build['#markup'] = $processed_content;
        $build['#allowed_tags'] = dcr_renderer_get_allowed_tags();
        break;

      case 'hardened':
        $base_url = $this->getBaseUrl($url);
        $result = dcr_renderer_process_hardened_content($content, $base_url);
        $build['#markup'] = $result['content'];
        $build['#allowed_tags'] = dcr_renderer_get_allowed_tags();

        // Attach extracted assets.
        if (!empty($result['assets'])) {
          $this->attachAssets($build, $result['assets']);
        }
        break;

      case 'full':
        // Return the full HTML document.
        $base_url = $this->getBaseUrl($url);
        $processed_content = dcr_renderer_process_proxied_content($content, $base_url);
        $build['#markup'] = $processed_content;
        $build['#allowed_tags'] = dcr_renderer_get_allowed_tags();
        break;

      default:
        $build['#markup'] = '<div class="dcr-error">Unknown processing type: ' . htmlspecialchars($processing_type) . '</div>';
    }

    return $build;
  }

  /**
   * Fetch content from URL or local path.
   *
   * @param string $url
   *   The URL or path to fetch from.
   *
   * @return string|false
   *   The content or FALSE on failure.
   */
  protected function fetchContent($url) {
    $logger = $this->loggerFactory->get('dcr_renderer');

    // Check if this is a local file path.
    if (!preg_match('/^https?:\/\//', $url)) {
      // Local file path.
      $file_system = \Drupal::service('file_system');
      
      // Handle relative paths starting with public://
      if (strpos($url, 'public://') === 0) {
        $real_path = $file_system->realpath($url);
      }
      // Handle absolute paths.
      elseif (strpos($url, '/') === 0) {
        $real_path = $url;
      }
      // Handle relative paths.
      else {
        $public_path = $file_system->realpath('public://');
        $real_path = $public_path . '/' . $url;
      }

      if (file_exists($real_path) && is_readable($real_path)) {
        $content = file_get_contents($real_path);
        $logger->info('Successfully read local file: @path', ['@path' => $real_path]);
        return $content;
      }
      else {
        $logger->error('Local file not found or not readable: @path', ['@path' => $real_path]);
        return FALSE;
      }
    }

    // Remote URL - use HTTP client.
    try {
      $client = \Drupal::httpClient();
      $response = $client->get($url, [
        'timeout' => 30,
        'headers' => [
          'User-Agent' => 'Drupal DCR Renderer Module',
        ],
      ]);

      if ($response->getStatusCode() === 200) {
        $content = $response->getBody()->getContents();
        $logger->info('Successfully fetched remote content from: @url', ['@url' => $url]);
        return $content;
      }
      else {
        $logger->error('HTTP error @status fetching: @url', [
          '@status' => $response->getStatusCode(),
          '@url' => $url,
        ]);
        return FALSE;
      }
    }
    catch (\Exception $e) {
      $logger->error('Exception fetching @url: @message', [
        '@url' => $url,
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Process content based on processing type.
   *
   * @param string $content
   *   The raw content.
   * @param string $url
   *   The original URL for context.
   * @param string $processing_type
   *   The processing type.
   *
   * @return string
   *   The processed content.
   */
  protected function processContent($content, $url, $processing_type) {
    switch ($processing_type) {
      case 'raw':
        return $content;

      case 'proxied':
        $base_url = $this->getBaseUrl($url);
        return dcr_renderer_process_proxied_content($content, $base_url);

      case 'hardened':
        $base_url = $this->getBaseUrl($url);
        $result = dcr_renderer_process_hardened_content($content, $base_url);
        return $result['content'];

      case 'full':
        $base_url = $this->getBaseUrl($url);
        return dcr_renderer_process_proxied_content($content, $base_url);

      default:
        return $content;
    }
  }

  /**
   * Extract base URL from a given URL.
   *
   * @param string $url
   *   The full URL.
   *
   * @return string
   *   The base URL (scheme + host + port).
   */
  protected function getBaseUrl($url) {
    // For local paths, use current site's base URL.
    if (!preg_match('/^https?:\/\//', $url)) {
      return \Drupal::request()->getSchemeAndHttpHost();
    }

    // For remote URLs, extract the base.
    $parsed = parse_url($url);
    if (!$parsed) {
      return \Drupal::request()->getSchemeAndHttpHost();
    }

    $base_url = $parsed['scheme'] . '://' . $parsed['host'];
    if (isset($parsed['port'])) {
      $base_url .= ':' . $parsed['port'];
    }

    return $base_url;
  }

  /**
   * Attach extracted assets to a render array.
   *
   * @param array &$build
   *   The render array to modify.
   * @param array $assets
   *   The extracted assets.
   */
  protected function attachAssets(array &$build, array $assets) {
    // Attach CSS files.
    foreach ($assets['css'] as $css_url) {
      $build['#attached']['html_head'][] = [
        [
          '#tag' => 'link',
          '#attributes' => [
            'rel' => 'stylesheet',
            'href' => $css_url,
          ],
        ],
        'dcr_css_' . md5($css_url),
      ];
    }

    // Attach JS files.
    foreach ($assets['js'] as $js_url) {
      $build['#attached']['html_head'][] = [
        [
          '#tag' => 'script',
          '#attributes' => [
            'src' => $js_url,
          ],
        ],
        'dcr_js_' . md5($js_url),
      ];
    }

    // Attach inline CSS.
    foreach ($assets['inline_css'] as $index => $inline_css) {
      $build['#attached']['html_head'][] = [
        [
          '#tag' => 'style',
          '#value' => $inline_css,
        ],
        'dcr_inline_css_' . $index,
      ];
    }

    // Attach inline JS.
    foreach ($assets['inline_js'] as $index => $inline_js) {
      $build['#attached']['html_head'][] = [
        [
          '#tag' => 'script',
          '#value' => $inline_js,
        ],
        'dcr_inline_js_' . $index,
      ];
    }
  }

}