<?php

declare(strict_types=1);

namespace Drupal\de_notifications_symfony_mailer\Plugin\NotificationType;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\de_notifications\NotificationsContextServiceInterface;
use Drupal\de_notifications\NotificationsSubscriberInterface;
use Drupal\de_notifications\NotificationsSubscriptionInterface;
use Drupal\de_notifications\Plugin\NotificationTypeBase;
use Drupal\symfony_mailer\Address;
use Drupal\symfony_mailer\EmailFactoryInterface;
use Drupal\symfony_mailer\EmailInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The Symfony mailer plugin.
 *
 * @NotificationType(
 *   id = "symfony_mail",
 *   label = @Translation("Symfony mailer"),
 * )
 */
class SymfonyMail extends NotificationTypeBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs the Symfony mailer plugin.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\de_notifications\NotificationsContextServiceInterface $contextService
   *   The notifications context service.
   * @param \Drupal\symfony_mailer\EmailFactoryInterface $emailFactory
   *   The email factory.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected readonly NotificationsContextServiceInterface $contextService,
    protected readonly EmailFactoryInterface $emailFactory,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('de_notifications.context'),
      $container->get('email_factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function sendConfirmation(NotificationsSubscriptionInterface $subscription): void {
    $this->contextService->setSubscription($subscription);

    $email = $this->composeMail('confirm', $subscription->get('langcode')->value);

    $email->setVariable('confirm_url', $this->contextService->getConfirmUrl())
      ->setVariable('unsubscribe_url', $this->contextService->getUnsubscribeUrl())
      ->setVariable('entity_title', $this->contextService->getLatestEntityTitle())
      ->setVariable('entity_url', $this->contextService->getLatestEntityUrl());

    $email->send();
  }

  /**
   * {@inheritdoc}
   */
  public function sendSubscriptionConfirmed(NotificationsSubscriptionInterface $subscription): void {
    $this->contextService->setSubscription($subscription);

    $email = $this->composeMail('subscription_confirmed', $subscription->get('langcode')->value);

    $email->setVariable('unsubscribe_url', $this->contextService->getUnsubscribeUrl())
      ->setVariable('entity_title', $this->contextService->getLatestEntityTitle())
      ->setVariable('entity_url', $this->contextService->getLatestEntityUrl());

    $email->send();
  }

  /**
   * {@inheritdoc}
   */
  public function sendAlreadySubscribed(NotificationsSubscriptionInterface $subscription): void {
    $this->contextService->setSubscription($subscription);

    $email = $this->composeMail('already_subscribed', $subscription->get('langcode')->value);

    $email->setVariable('confirm_url', $this->contextService->getConfirmUrl())
      ->setVariable('unsubscribe_url', $this->contextService->getUnsubscribeUrl())
      ->setVariable('entity_title', $this->contextService->getLatestEntityTitle())
      ->setVariable('entity_url', $this->contextService->getLatestEntityUrl());

    $email->send();
  }

  /**
   * {@inheritdoc}
   */
  public function sendSubscriptionOverview(NotificationsSubscriberInterface $subscriber, string $langcode): void {
    $this->contextService->setSubscriber($subscriber);

    $email = $this->composeMail('subscription_overview', $langcode);

    $email->setVariable('subscriptions', $this->contextService->getSubscriptions());

    $email->send();
  }

  /**
   * {@inheritdoc}
   */
  public function sendEntityNotification(NotificationsSubscriptionInterface $subscription, string $entity_title, string $changes): void {
    $this->contextService->setSubscription($subscription);

    $email = $this->composeMail('entity_notification', $subscription->get('langcode')->value);

    $email->setVariable('confirm_url', $this->contextService->getConfirmUrl())
      ->setVariable('unsubscribe_url', $this->contextService->getUnsubscribeUrl())
      ->setVariable('entity_title', $entity_title)
      ->setVariable('entity_url', $this->contextService->getLatestEntityUrl())
      ->setVariable('changes', $changes);

    $email->send();
  }

  /**
   * {@inheritdoc}
   */
  public function sendArchived(NotificationsSubscriptionInterface $subscription): void {
    $this->contextService->setSubscription($subscription);

    $email = $this->composeMail('archived', $subscription->get('langcode')->value);

    $email->setVariable('confirm_url', $this->contextService->getConfirmUrl())
      ->setVariable('unsubscribe_url', $this->contextService->getUnsubscribeUrl())
      ->setVariable('entity_title', $this->contextService->getLatestEntityTitle())
      ->setVariable('entity_url', $this->contextService->getLatestEntityUrl());

    $email->send();
  }

  /**
   * Compose email with common variables for all subtypes of mails.
   *
   * @param string $sub_type
   *   The Symfony Mailer subtype of the email.
   * @param string $langcode
   *   The language of the email.
   *
   * @return \Drupal\symfony_mailer\EmailInterface
   *   The email to send.
   *
   * @throws \Drupal\Core\TypedData\Exception\MissingDataException
   * @throws \Exception
   */
  protected function composeMail(string $sub_type, string $langcode): EmailInterface {
    $this->contextService->setLanguage($langcode);
    $address = new Address($this->contextService->getSubscriberEmail(), NULL, $langcode);
    return $this->emailFactory->newTypedEmail('de_notifications_mailer', $sub_type)
      ->setTo($address)
      ->setVariable('unsubscribe_all_url', $this->contextService->getUnsubscribeAllUrl())
      ->setVariable('request_subscription_overview_url', $this->contextService->getRequestSubscriptionOverviewUrl());
  }

}
