<?php

declare(strict_types=1);

namespace Drupal\de_notifications\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Language\LanguageInterface;
use Drupal\de_notifications\Exception\NotificationsException;
use Drupal\de_notifications\NotificationsSubscriptionHelperInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Class NotificationsSubscriptionController.
 *
 * Controls API endpoints for subscribing and unsubscribing to entities.
 */
class NotificationsSubscriptionController extends ControllerBase {

  /**
   * The NotificationsSubscriptionController constructor.
   *
   * @param \Drupal\de_notifications\NotificationsSubscriptionHelperInterface $subscriptionHelper
   *   The subscription helper service.
   */
  public function __construct(protected readonly NotificationsSubscriptionHelperInterface $subscriptionHelper) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('de_notifications.subscription_helper')
    );
  }

  /**
   * Subscribe a user to a given entity.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   Response
   */
  public function subscribe(Request $request): JsonResponse {
    $email = (string) $request->get('email');
    $entity_type = (string) $request->get('entity_type');
    $eid = (string) $request->get('eid');
    $langcode = $this->languageManager()->getCurrentLanguage(LanguageInterface::TYPE_CONTENT)->getId();
    $ip_address = $request->headers->get('X-Client-Ip');

    if (empty($email)) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Parameter email is required.',
      ], 400);
    }

    if (empty($entity_type)) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Parameter entity_type is required.',
      ], 400);
    }

    if (empty($eid)) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Parameter eid is required.',
      ], 400);
    }

    if (empty($ip_address)) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Header X-Client-Ip is required.',
      ], 400);
    }

    try {
      $data = $this->subscriptionHelper->subscribe($email, $entity_type, $eid, $langcode, $ip_address);
    }
    catch (NotificationsException $exception) {
      return new JsonResponse([
        'status' => 'error',
        'message' => $exception->getMessage(),
      ], $exception->getCode() ?? 500);
    }
    catch (\Throwable) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Something went wrong. Please try again later.',
      ], 500);
    }

    return new JsonResponse([
      'status' => 'success',
      'message' => 'User was successfully subscribed.',
      'data' => $data,
    ]);
  }

  /**
   * Confirm a subscription.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   Response
   */
  public function confirm(Request $request): JsonResponse {
    $token = (string) $request->get('token');

    if (empty($token)) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Parameter token is required.',
      ], 400);
    }

    try {
      $data = $this->subscriptionHelper->confirm($token);
    }
    catch (NotificationsException $exception) {
      return new JsonResponse([
        'status' => 'error',
        'message' => $exception->getMessage(),
      ], $exception->getCode() ?? 500);
    }
    catch (\Throwable) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Something went wrong. Please try again later.',
      ], 500);
    }

    return new JsonResponse([
      'status' => 'success',
      'message' => 'Subscription was ' . ($data['alreadyConfirmed'] ? 'already' : 'successfully') . ' confirmed.',
      'data' => $data,
    ]);
  }

  /**
   * Unsubscribe from a single entity by subscription token.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   Response
   */
  public function unsubscribe(Request $request): JsonResponse {
    $token = (string) $request->get('token');

    if (empty($token)) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Parameter token is required.',
      ], 400);
    }

    try {
      $data = $this->subscriptionHelper->unsubscribe($token);
    }
    catch (NotificationsException $exception) {
      return new JsonResponse([
        'status' => 'error',
        'message' => $exception->getMessage(),
      ], $exception->getCode() ?? 500);
    }
    catch (\Throwable) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Something went wrong. Please try again later.',
      ], 500);
    }

    return new JsonResponse([
      'status' => 'success',
      'message' => 'User was ' . ($data['alreadyUnsubscribed'] ? 'already' : 'successfully') . ' unsubscribed.',
      'data' => $data,
    ]);
  }

  /**
   * Unsubscribe from all entities by subscriber token.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   Response
   */
  public function unsubscribeAll(Request $request): JsonResponse {
    $token = (string) $request->get('token');

    if (empty($token)) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Parameter token is required',
      ], 400);
    }

    try {
      $data = $this->subscriptionHelper->unsubscribeAll($token);
    }
    catch (NotificationsException $exception) {
      return new JsonResponse([
        'status' => 'error',
        'message' => $exception->getMessage(),
      ], $exception->getCode() ?? 500);
    }
    catch (\Throwable) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Something went wrong. Please try again later.',
      ], 500);
    }

    return new JsonResponse([
      'status' => 'success',
      'message' => 'User was ' . ($data['alreadyUnsubscribed'] ? 'already' : 'successfully') . ' unsubscribed from all entities.',
      'data' => $data,
    ]);
  }

  /**
   * Send subscription overview to subscriber.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   Response
   */
  public function requestSubscriptionOverview(Request $request): JsonResponse {
    $token = (string) $request->get('token');
    $langcode = $this->languageManager()->getCurrentLanguage(LanguageInterface::TYPE_CONTENT)->getId();

    if (empty($token)) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Parameter token is required',
      ], 400);
    }

    try {
      $this->subscriptionHelper->requestSubscriptionOverview($token, $langcode);
    }
    catch (NotificationsException $exception) {
      return new JsonResponse([
        'status' => 'error',
        'message' => $exception->getMessage(),
      ], $exception->getCode() ?? 500);
    }
    catch (\Throwable) {
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Something went wrong. Please try again later.',
      ], 500);
    }

    return new JsonResponse([
      'status' => 'success',
      'message' => 'Subscription overview was sent.',
    ]);
  }

}
