<?php

declare(strict_types=1);

namespace Drupal\de_notifications\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\de_notifications\NotificationsSubscriberInterface;

/**
 * Defines the notifications subscriber entity class.
 *
 * @ContentEntityType(
 *   id = "de_notifications_subscriber",
 *   label = @Translation("Notifications subscriber"),
 *   label_collection = @Translation("Notifications subscribers"),
 *   label_singular = @Translation("notifications subscriber"),
 *   label_plural = @Translation("notifications subscribers"),
 *   label_count = @PluralTranslation(
 *     singular = "@count notifications subscribers",
 *     plural = "@count notifications subscribers",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\de_notifications\NotificationsSubscriberListBuilder",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "access" = "Drupal\de_notifications\NotificationsSubscriberAccessControlHandler",
 *     "form" = {
 *       "add" = "Drupal\de_notifications\Form\NotificationsSubscriberForm",
 *       "edit" = "Drupal\de_notifications\Form\NotificationsSubscriberForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *       "delete-multiple-confirm" = "Drupal\Core\Entity\Form\DeleteMultipleForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\de_notifications\Routing\NotificationsSubscriberHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "de_notifications_subscriber",
 *   admin_permission = "administer de_notifications_subscriber",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "email",
 *     "uuid" = "uuid",
 *   },
 *   links = {
 *     "add-form" = "/de_notifications/subscriber/add",
 *     "canonical" = "/de_notifications/subscriber/{de_notifications_subscriber}",
 *     "edit-form" = "/de_notifications/{de_notifications_subscriber}/edit",
 *     "delete-form" = "/de_notifications/{de_notifications_subscriber}/delete",
 *     "delete-multiple-form" = "/de_notifications/subscriber/delete-multiple",
 *   },
 *   field_ui_base_route = "entity.de_notifications_subscriber.settings",
 * )
 */
class NotificationsSubscriber extends ContentEntityBase implements NotificationsSubscriberInterface {

  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {

    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['email'] = BaseFieldDefinition::create('email')
      ->setLabel(t('Email'))
      ->setDescription(t('The email of the subscriber.'))
      ->setDefaultValue('')
      ->setRequired(TRUE)
      ->addConstraint('UniqueField')
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'email_default',
        'weight' => 0,
      ]);

    $fields['ip_address'] = BaseFieldDefinition::create('string')
      ->setLabel(t('IP address'))
      ->setDescription(t('The IP address of the subscriber.'))
      ->setRequired(TRUE)
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 3,
      ]);

    $fields['bounce_count'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Bounce count'))
      ->setDefaultValue(0)
      ->setDescription(t('The bounce count of the subscriber.'))
      ->setSetting('unsigned', TRUE)
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'number',
        'weight' => 4,
      ]);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Authored on'))
      ->setDescription(t('The time that the notifications subscriber was created.'))
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 5,
      ]);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the notifications subscriber was last edited.'));

    return $fields;
  }

  /**
   * Implementation of the magic __toString() method.
   *
   * @return string
   *   The unique mail address of the subscriber
   */
  public function __toString() {
    return $this->get('email')->value;
  }

}
