<?php

declare(strict_types=1);

namespace Drupal\de_notifications\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\de_notifications\NotificationsSubscriptionInterface;

/**
 * Defines the notifications subscription entity class.
 *
 * @ContentEntityType(
 *   id = "de_notifications_subscription",
 *   label = @Translation("Notifications subscription"),
 *   label_collection = @Translation("Notifications subscriptions"),
 *   label_singular = @Translation("notifications subscription"),
 *   label_plural = @Translation("notifications subscriptions"),
 *   label_count = @PluralTranslation(
 *     singular = "@count notifications subscriptions",
 *     plural = "@count notifications subscriptions",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\de_notifications\NotificationsSubscriptionListBuilder",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "access" = "Drupal\de_notifications\NotificationsSubscriptionAccessControlHandler",
 *     "form" = {
 *       "add" = "Drupal\de_notifications\Form\NotificationsSubscriptionForm",
 *       "edit" = "Drupal\de_notifications\Form\NotificationsSubscriptionForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *       "delete-multiple-confirm" = "Drupal\Core\Entity\Form\DeleteMultipleForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\de_notifications\Routing\NotificationsSubscriptionHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "de_notifications_subscription",
 *   admin_permission = "administer de_notifications_subscription",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "id",
 *     "uuid" = "uuid",
 *   },
 *   links = {
 *     "add-form" = "/de_notifications/subscription/add",
 *     "canonical" = "/de_notifications/subscriber/{de_notifications_subscription}",
 *     "edit-form" = "/de_notifications/subscription/{de_notifications_subscription}/edit",
 *     "delete-form" = "/de_notifications/subscription/{de_notifications_subscription}/delete",
 *     "delete-multiple-form" = "/de_notifications/subscription/delete-multiple",
 *   },
 *   field_ui_base_route = "entity.de_notifications_subscription.settings",
 * )
 */
class NotificationsSubscription extends ContentEntityBase implements NotificationsSubscriptionInterface {

  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);

    $target_entity_bundles = \Drupal::state()->get('target_entity_bundles') ?? [];

    $fields['subscriber'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Subscriber'))
      ->setDescription(t('The subscriber of the subscription.'))
      ->setSetting('target_type', 'de_notifications_subscriber')
      ->setRequired(TRUE)
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 0,
      ]);

    $fields['entity'] = BaseFieldDefinition::create('dynamic_entity_reference')
      ->setLabel(t('Entity'))
      ->setDescription(t('The subscribed entity.'))
      ->setSetting('entity_type_ids', array_keys($target_entity_bundles))
      ->setSetting('exclude_entity_types', FALSE)
      ->setSettings(array_reduce(array_keys($target_entity_bundles), function ($carry, $entity_type) use ($target_entity_bundles) {
        $carry[$entity_type] = [
          'handler' => 'default',
          'handler_settings' => [
            'target_bundles' => $target_entity_bundles[$entity_type],
          ],
        ];
        return $carry;
      }, [])
      )
      ->setRequired(TRUE)
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'dynamic_entity_reference_default',
        'weight' => 1,
      ]);

    $fields['langcode'] = BaseFieldDefinition::create('language')
      ->setLabel(t('Language'))
      ->setDescription(t('The subscribed entity language.'))
      ->setRequired(TRUE)
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'language_select',
        'weight' => 2,
        'settings' => [
          'include_locked' => FALSE,
        ],
      ]);

    $fields['is_confirmed'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Is confirmed'))
      ->setDescription(t('The subscription is confirmed.'))
      ->setDefaultValue(FALSE)
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'weight' => 3,
      ]);

    $fields['last_confirmation_sent'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Last confirmation sent'))
      ->setDescription(t('The time that the last confirmation email was sent for this subscription.'))
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 4,
      ]);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Authored on'))
      ->setDescription(t('The time that the notifications subscription was created.'))
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 4,
      ]);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the notifications subscription was last edited.'));

    return $fields;
  }

}
