<?php

declare(strict_types=1);

namespace Drupal\de_notifications\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\de_notifications\Plugin\NotificationTypeManagerInterface;
use Psr\Container\ContainerInterface;

/**
 * Notifications General Configuration form.
 */
class NotificationsSettingsForm extends ConfigFormBase implements ContainerInjectionInterface {

  const SETTINGS = 'de_notifications.settings';

  /**
   * Constructs a NotificationsSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\de_notifications\Plugin\NotificationTypeManagerInterface $notificationTypeManager
   *   The notification type plugin manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typedConfigManager,
    protected readonly NotificationTypeManagerInterface $notificationTypeManager,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('plugin.manager.notification_type')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'de_notifications_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [self::SETTINGS];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['notification_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Notification type'),
      '#config_target' => self::SETTINGS . ':notification_type',
      '#options' => array_merge(['' => $this->t('None')], array_column($this->notificationTypeManager->getDefinitions(), 'label', 'id')),
      '#description' => $this->t('Select the notification type to use for notifications.'),
      '#required' => TRUE,
    ];

    $form['urls'] = [
      '#type' => 'fieldset',
      '#title' => $this->t("Frontend URL's"),
      '#collapsible' => FALSE,
      '#collapsed' => FALSE,
    ];

    $form['urls']['frontend_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Frontend base URL'),
      '#config_target' => self::SETTINGS . ':frontend_url',
      '#description' => $this->t('Frontend base URL for all actions (e.g., https://example.com).'),
      '#required' => TRUE,
      '#pattern' => '.+(?<!/)$',
    ];

    $form['urls']['confirm_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subscription confirm relative path'),
      '#config_target' => self::SETTINGS . ':confirm_path',
      '#description' => $this->t('Frontend relative path starting with / (e.g., /confirm).'),
      '#required' => TRUE,
      '#pattern' => '^/.+(?<!/)$',
    ];

    $form['urls']['unsubscribe_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Unsubscribe relative path'),
      '#config_target' => self::SETTINGS . ':unsubscribe_path',
      '#description' => $this->t('Frontend relative path starting with / (e.g., /unsubscribe).'),
      '#required' => TRUE,
      '#pattern' => '^/.+(?<!/)$',
    ];

    $form['urls']['unsubscribe_all_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Unsubscribe all relative path'),
      '#config_target' => self::SETTINGS . ':unsubscribe_all_path',
      '#description' => $this->t('Frontend relative path starting with / (e.g., /unsubscribe-all).'),
      '#required' => TRUE,
      '#pattern' => '^/.+(?<!/)$',
    ];

    $form['urls']['request_subscription_overview_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Request subscription overview relative path'),
      '#config_target' => self::SETTINGS . ':request_subscription_overview_path',
      '#description' => $this->t('Frontend relative path starting with / (e.g., /subscription-overview).'),
      '#required' => TRUE,
      '#pattern' => '^/.+(?<!/)$',
    ];

    $form['urls']['notification_link_query'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Query parameters'),
      '#config_target' => self::SETTINGS . ':notification_link_query',
      '#description' => $this->t('Optional: Add query parameters to the entity link in a notification (e.g. utm_source=notification&utm_medium=email).'),
      '#required' => FALSE,
      '#pattern' => '^([\w-]+(=[\w-]*)?(&[\w-]+(=[\w-]*)?)*)?$',
    ];

    $form['ttl'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Time to live for tokens'),
      '#collapsible' => FALSE,
      '#collapsed' => FALSE,
    ];

    $form['ttl']['ttl_confirm'] = [
      '#type' => 'number',
      '#title' => $this->t('Time to live for confirm url token'),
      '#config_target' => self::SETTINGS . ':ttl_confirm',
      '#description' => $this->t('Time to live for confirm url tokens in seconds. (Default: 24 hours)'),
      '#required' => TRUE,
    ];

    $form['ttl']['ttl_generic'] = [
      '#type' => 'number',
      '#title' => $this->t('Time to live for other tokens'),
      '#config_target' => self::SETTINGS . ':ttl_generic',
      '#description' => $this->t('Time to live for other tokens in seconds. (Default: 1 year)'),
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config(self::SETTINGS)
      ->set('frontend_url', $form_state->getValue('frontend_url'))
      ->set('confirm_path', $form_state->getValue('confirm_path'))
      ->set('unsubscribe_path', $form_state->getValue('unsubscribe_path'))
      ->set('unsubscribe_all_path', $form_state->getValue('unsubscribe_all_path'))
      ->set('request_subscription_overview_path', $form_state->getValue('request_subscription_overview_path'))
      ->set('ttl_confirm', $form_state->getValue('ttl_confirm'))
      ->set('ttl_generic', $form_state->getValue('ttl_generic'))
      ->set('notification_type', $form_state->getValue('notification_type'))
      ->save();
    $this->messenger()->addStatus($this->t('The configuration has been updated.'));
  }

}
