<?php

declare(strict_types=1);

namespace Drupal\de_notifications;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Url;

/**
 * The Notifications Context Service.
 */
class NotificationsContextService implements NotificationsContextServiceInterface {

  /**
   * The Subscription.
   *
   * @var \Drupal\de_notifications\NotificationsSubscriptionInterface
   */
  private NotificationsSubscriptionInterface $subscription;

  /**
   * The Subscriber.
   *
   * @var \Drupal\de_notifications\NotificationsSubscriberInterface
   */
  private NotificationsSubscriberInterface $subscriber;

  /**
   * The Entity.
   *
   * @var \Drupal\Core\Entity\EntityInterface
   */
  private EntityInterface $entity;

  /**
   * Constructs a NotificationsContextService object.
   */
  public function __construct(
    protected readonly NotificationsTokenServiceInterface $tokenService,
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly LanguageManagerInterface $languageManager,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
  }

  /**
   * {@inheritDoc}
   */
  public function setSubscription(NotificationsSubscriptionInterface $subscription): void {
    $this->subscription = $subscription;
    $this->subscriber = $subscription->get('subscriber')->entity;

    $entity = $subscription->get('entity')->entity;
    if (method_exists($entity, 'getTranslation')) {
      $entity = $entity->getTranslation($subscription->get('langcode')->value);
    }

    $this->entity = $entity;
  }

  /**
   * {@inheritDoc}
   */
  public function setSubscriber(NotificationsSubscriberInterface $subscriber): void {
    $this->subscriber = $subscriber;
  }

  /**
   * {@inheritDoc}
   */
  public function setLanguage(string $langcode): void {
    $language = $this->languageManager->getLanguage($langcode);
    if ($language) {
      $this->languageManager->setConfigOverrideLanguage($language);
    }
  }

  /**
   * {@inheritDoc}
   */
  public function getLatestEntityTitle(): string {
    if (!isset($this->subscription)) {
      throw new \Exception('Missing $subscription variable');
    }

    return $this->entity->label();
  }

  /**
   * {@inheritDoc}
   */
  public function getLatestEntityUrl(): string {
    if (!isset($this->subscription)) {
      throw new \Exception('Missing $subscription variable');
    }

    $config = $this->configFactory->get('de_notifications.settings');
    $options = [
      'absolute' => TRUE,
      'base_url' => $config->get('frontend_url'),
    ];

    $queryInput = $config->get('notification_link_query');
    if (!empty($queryInput)) {
      parse_str($queryInput, $query);
      $options['query'] = $query;
    }
    return $this->entity->toUrl('canonical', $options)->toString();
  }

  /**
   * {@inheritDoc}
   */
  public function getSubscriberEmail(): string {
    if (!isset($this->subscriber)) {
      throw new \Exception('Missing $subscriber variable');
    }

    return $this->subscriber->get('email')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmUrl(): string {
    if (!isset($this->subscription)) {
      throw new \Exception('Missing $subscription variable');
    }
    return $this->getUrlWithToken('confirm_path', $this->subscription->uuid());
  }

  /**
   * {@inheritdoc}
   */
  public function getUnsubscribeUrl(): string {
    if (!isset($this->subscription)) {
      throw new \Exception('Missing $subscription variable');
    }
    return $this->getUrlWithToken('unsubscribe_path', $this->subscription->uuid());
  }

  /**
   * {@inheritdoc}
   */
  public function getUnsubscribeAllUrl(): string {
    if (!isset($this->subscriber)) {
      throw new \Exception('Missing $subscriber variable');
    }
    return $this->getUrlWithToken('unsubscribe_all_path', $this->subscriber->uuid());
  }

  /**
   * {@inheritdoc}
   */
  public function getRequestSubscriptionOverviewUrl(): string {
    if (!isset($this->subscriber)) {
      throw new \Exception('Missing $subscriber variable');
    }
    return $this->getUrlWithToken('request_subscription_overview_path', $this->subscriber->uuid());
  }

  /**
   * {@inheritDoc}
   */
  public function getSubscriptions(): array {
    if (!isset($this->subscriber)) {
      throw new \Exception('Missing $subscriber variable');
    }
    $subscriptions = $this->entityTypeManager
      ->getStorage('de_notifications_subscription')
      ->loadByProperties([
        'subscriber' => $this->subscriber->id(),
        'is_confirmed' => TRUE,
      ]);

    $formattedSubscriptions = [];
    foreach ($subscriptions as $subscription) {
      $entity = $subscription->get('entity')->entity;
      if (method_exists($entity, 'getTranslation')) {
        $entity = $entity->getTranslation($subscription->get('langcode')->value);
      }
      $formattedSubscriptions[] = [
        'title' => $entity->get('title')->value,
        'unsubscribe_url' => $this->getUrlWithToken('unsubscribe_path', $subscription->uuid()),
      ];
    }

    return $formattedSubscriptions;
  }

  /**
   * Returns url string with token parameter included.
   *
   * @param string $url_name
   *   The config name of the url.
   * @param string $uuid
   *   Uuid of the subscriber or subscription.
   *
   * @return string
   *   The url.
   *
   * @throws \Exception
   */
  protected function getUrlWithToken(string $url_name, string $uuid): string {
    $config = $this->configFactory->get('de_notifications.settings');

    $params = [
      't' => $this->tokenService->generateToken(
        $uuid,
        $url_name === 'confirm_path' ? $config->get('ttl_confirm') : $config->get('ttl_generic')
      ),
    ];

    $url = Url::fromUri(
      $config->get('frontend_url') . $config->get($url_name),
      [
        'query' => $params,
        'absolute' => TRUE,
      ]
    );
    return $url->toString();
  }

}
