<?php

declare(strict_types=1);

namespace Drupal\de_notifications;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\de_notifications\Exception\NotificationsException;
use Drupal\de_notifications\Plugin\NotificationTypeInterface;
use Drupal\de_notifications\Plugin\NotificationTypeManagerInterface;

/**
 * Class NotificationsSubscriptionHelper.
 *
 * Manages subscribing and unsubscribing to entities.
 */
class NotificationsSubscriptionHelper implements NotificationsSubscriptionHelperInterface {

  /**
   * The NotificationsSubscriptionHelper constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The Entity type manager container.
   * @param \Drupal\de_notifications\NotificationsTokenServiceInterface $tokenService
   *   The Custom token service to handle all token actions.
   * @param \Drupal\de_notifications\Plugin\NotificationTypeManagerInterface $notificationTypeManager
   *   The notification type plugin manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time.
   */
  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly NotificationsTokenServiceInterface $tokenService,
    protected readonly NotificationTypeManagerInterface $notificationTypeManager,
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly TimeInterface $time,
  ) {}

  /**
   * {@inheritDoc}
   */
  public function subscribe(string $email, string $entity_type, string $eid, string $langcode, string $ip_address): array {
    try {
      $entity_storage = $this->entityTypeManager->getStorage($entity_type);
    }
    catch (PluginNotFoundException) {
      throw new NotificationsException("Entity type {$entity_type} not found.", 404);
    }

    /** @var \Drupal\Core\Entity\EntityInterface $entity */
    $entity = $entity_storage->load($eid);

    if (!$entity) {
      throw new NotificationsException("The entity type {$entity_type} with id {$eid} does not exist.", 400);
    }

    if ($entity->language()->getId() !== $langcode) {
      if ($entity->isTranslatable() && $entity->hasTranslation($langcode)) {
        $entity = $entity->getTranslation($langcode);
      }
      else {
        throw new NotificationsException("The langcode {$langcode} with entity id {$eid} does not exist.", 400);
      }
    }

    if (!$this->subscriptionEnabled($entity)) {
      if ($entity->isTranslatable()) {
        throw new NotificationsException("It is not allowed to subscribe to notifications on {$entity_type} with id {$eid} and langcode {$langcode}.", 403);
      }
      throw new NotificationsException("It is not allowed to subscribe to notifications on {$entity_type} with id {$eid}.", 403);
    }

    $notification_type = $this->getNotificationType();

    $subscriber_storage = $this->entityTypeManager->getStorage('de_notifications_subscriber');
    $subscription_storage = $this->entityTypeManager->getStorage('de_notifications_subscription');
    /** @var \Drupal\de_notifications\NotificationsSubscriberInterface $subscriber */
    $subscriber = current($subscriber_storage->loadByProperties([
      'email' => $email,
    ]));

    if ($subscriber) {
      /** @var \Drupal\de_notifications\NotificationsSubscriptionInterface $subscription */
      $subscription = current($subscription_storage->loadByProperties([
        'subscriber' => $subscriber->id(),
        'entity' => [
          'target_type' => $entity_type,
          'target_id' => $eid,
        ],
        'langcode' => $langcode,
      ]));

      if ($subscription) {
        if ($subscription->get('is_confirmed')->value) {
          // User is already subscribed, send already subscribed notification.
          $notification_type->sendAlreadySubscribed($subscription);
        }
        else {
          // User has not confirmed subscription, resend confirm notification.
          $subscription->set('last_confirmation_sent', $this->time->getRequestTime());
          $subscription->save();
          $notification_type->sendConfirmation($subscription);
        }

        return [
          ...$this->getEntityData($subscription),
        ];
      }
    }
    else {
      // Create subscriber if doesn't exist yet.
      /** @var \Drupal\de_notifications\NotificationsSubscriberInterface $subscriber */
      $subscriber = $subscriber_storage->create([
        'email' => $email,
        'ip_address' => $ip_address,
      ]);

      $this->validate($subscriber);
      $subscriber->save();
    }

    // Subscribe if not subscribed yet.
    /** @var \Drupal\de_notifications\NotificationsSubscriptionInterface $subscription */
    $subscription = $subscription_storage->create([
      'subscriber' => $subscriber->id(),
      'entity' => [
        'target_type' => $entity_type,
        'target_id' => $eid,
      ],
      'langcode' => $langcode,
    ]);

    $this->validate($subscription);
    $subscription->save();
    $notification_type->sendConfirmation($subscription);
    return [
      ...$this->getEntityData($subscription),
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function confirm(string $token): array {
    $uuid = $this->tokenService->validateToken($token);
    /** @var \Drupal\de_notifications\NotificationsSubscriptionInterface $subscription */
    $subscription = current($this->entityTypeManager->getStorage('de_notifications_subscription')->loadByProperties([
      'uuid' => $uuid,
    ]));

    if (!$subscription) {
      throw new NotificationsException('Cannot confirm, because subscription does not exist (anymore)', 404);
    }

    if ($subscription->get('is_confirmed')->value) {
      return [
        'alreadyConfirmed' => TRUE,
        ...$this->getEntityData($subscription),
      ];
    }

    $subscription->set('is_confirmed', TRUE);
    $subscription->save();
    $this->getNotificationType()->sendSubscriptionConfirmed($subscription);

    return [
      'alreadyConfirmed' => FALSE,
      ...$this->getEntityData($subscription),
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function unsubscribe(string $token): array {
    $uuid = $this->tokenService->validateToken($token);
    /** @var \Drupal\de_notifications\NotificationsSubscriptionInterface $subscription */
    $subscription = current($this->entityTypeManager->getStorage('de_notifications_subscription')->loadByProperties([
      'uuid' => $uuid,
    ]));

    if (!$subscription) {
      return [
        'alreadyUnsubscribed' => TRUE,
      ];
    }

    /** @var \Drupal\de_notifications\NotificationsSubscriberInterface $subscriber */
    $subscriber = $subscription->get('subscriber')->entity;

    $subscription->delete();
    $this->cleanSubscriber($subscriber);
    return [
      'alreadyUnsubscribed' => FALSE,
      ...$this->getEntityData($subscription),
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function unsubscribeAll(string $token): array {
    $uuid = $this->tokenService->validateToken($token);
    /** @var \Drupal\de_notifications\NotificationsSubscriberInterface $subscriber */
    $subscriber = current($this->entityTypeManager->getStorage('de_notifications_subscriber')->loadByProperties([
      'uuid' => $uuid,
    ]));

    if (!$subscriber) {
      return [
        'alreadyUnsubscribed' => TRUE,
      ];
    }

    $this->cleanSubscriber($subscriber, TRUE);
    return [
      'alreadyUnsubscribed' => FALSE,
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function requestSubscriptionOverview(string $token, string $langcode): void {
    $uuid = $this->tokenService->validateToken($token);
    /** @var \Drupal\de_notifications\NotificationsSubscriberInterface $subscriber */
    $subscriber = current($this->entityTypeManager->getStorage('de_notifications_subscriber')->loadByProperties([
      'uuid' => $uuid,
    ]));

    if (!$subscriber) {
      throw new NotificationsException('Cannot send subscription overview because subscriber has unsubscribed from all entities', 404);
    }

    $notification_type = $this->getNotificationType();
    $notification_type->sendSubscriptionOverview($subscriber, $langcode);
  }

  /**
   * Validates an entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   Subscriber or subscription entity to validate.
   *
   * @throws \Drupal\de_notifications\Exception\NotificationsException
   */
  protected function validate(EntityInterface $entity): void {
    $violations = $entity->validate();
    if (count($violations)) {
      $violation_messages = [];
      foreach ($violations as $violation) {
        $violation_messages[] = 'Field ' . $violation->getPropertyPath() . ': ' . $violation->getMessage();
      }
      throw new NotificationsException(implode(', ', $violation_messages), 400);
    }
  }

  /**
   * Checks if users are allowed to subscribe to an entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   Entity that is requested to subscribe to.
   *
   * @return bool
   *   Returns true if subscribing is allowed on the entity, otherwise false
   */
  protected function subscriptionEnabled(EntityInterface $entity): bool {
    if (!method_exists($entity, 'getFields')) {
      return FALSE;
    }

    $fields = $entity->getFields();
    foreach ($fields as $field) {
      if ($field->getFieldDefinition()
        ->getType() === 'notification_settings' && $entity->get($field->getName())->subscription_enabled) {
        return TRUE;
      }
    }
    return FALSE;
  }

  /**
   * Clean up a subscriber if they have no current subscriptions.
   *
   * @param \Drupal\de_notifications\NotificationsSubscriberInterface $subscriber
   *   The subscriber to delete if they have no current subscriptions.
   * @param bool $cleanSubscriptions
   *   Whether to delete all subscriptions related to the subscriber.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  protected function cleanSubscriber(NotificationsSubscriberInterface $subscriber, bool $cleanSubscriptions = FALSE): void {
    $subscription_storage = $this->entityTypeManager->getStorage('de_notifications_subscription');
    $subscriptions = $subscription_storage->loadByProperties([
      'subscriber' => $subscriber->id(),
    ]);
    if ($cleanSubscriptions) {
      $subscription_storage->delete($subscriptions);
      $subscriber->delete();
    }
    elseif (empty($subscriptions)) {
      $subscriber->delete();
    }
  }

  /**
   * Get notification type.
   *
   * @return \Drupal\de_notifications\Plugin\NotificationTypeInterface
   *   Returns the notification type instance.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   * @throws \Drupal\de_notifications\Exception\NotificationsException
   */
  protected function getNotificationType(): NotificationTypeInterface {
    $config = $this->configFactory->get('de_notifications.settings');
    $plugin_id = $config->get('notification_type');
    if (!$plugin_id) {
      throw new NotificationsException('There is no notification type set', 409);
    }
    if (!$this->notificationTypeManager->hasDefinition($plugin_id)) {
      throw new NotificationsException('Notification type ' . $plugin_id . ' does not exist', 409);
    }

    /** @var \Drupal\de_notifications\Plugin\NotificationTypeInterface $instance */
    $instance = $this->notificationTypeManager->createInstance($plugin_id);

    return $instance;
  }

  /**
   * Get additional data about the linked entity of a subscription.
   *
   * @param \Drupal\de_notifications\NotificationsSubscriptionInterface $subscription
   *   The subscription.
   *
   * @return array
   *   An array containing the entity data or
   *   an empty array if the client doesn't have permission.
   *
   * @throws \Drupal\Core\TypedData\Exception\MissingDataException
   */
  protected function getEntityData(NotificationsSubscriptionInterface $subscription): array {
    /** @var \Drupal\Core\Entity\EntityInterface $entity */
    $entity = $subscription->get('entity')->entity;
    $langcode = $subscription->get('langcode')->value;

    if ($entity->language()->getId() !== $langcode) {
      $entity = $entity->getTranslation($langcode);
    }

    if ($entity->access('view')) {
      return ['entityLabel' => $entity->label()];
    }

    return [];
  }

}
