<?php

namespace Drupal\de_notifications;

/**
 * Provides an interface for the NotificationsSubscriptionHelper.
 */
interface NotificationsSubscriptionHelperInterface {

  /**
   * Subscribe a user to a given entity.
   *
   * @param string $email
   *   Email address of the subscriber.
   * @param string $entity_type
   *   Entity type of the entity to subscribe to.
   * @param string $eid
   *   ID of the entity to subscribe to.
   * @param string $langcode
   *   The langcode of the subscriber.
   * @param string $ip_address
   *   Ip address of the subscriber.
   *
   * @return array
   *   Data array.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Drupal\Core\TypedData\Exception\MissingDataException
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   * @throws \Drupal\de_notifications\Exception\NotificationsException
   */
  public function subscribe(string $email, string $entity_type, string $eid, string $langcode, string $ip_address): array;

  /**
   * Confirm a subscription.
   *
   * @param string $token
   *   Token of the subscription.
   *
   * @return array
   *   Data array.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Drupal\de_notifications\Exception\NotificationsException
   */
  public function confirm(string $token): array;

  /**
   * Unsubscribe from a single entity by subscription token.
   *
   * @param string $token
   *   Token of the subscription.
   *
   * @return array
   *   Data array.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\TypedData\Exception\MissingDataException
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Drupal\de_notifications\Exception\NotificationsException
   */
  public function unsubscribe(string $token): array;

  /**
   * Unsubscribe from all entities by subscriber token.
   *
   * @param string $token
   *   Token of the subscriber.
   *
   * @return array
   *   Data array.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Drupal\de_notifications\Exception\NotificationsException
   */
  public function unsubscribeAll(string $token): array;

  /**
   * Request subscription overview by subscriber token and langcode.
   *
   * @param string $token
   *   Token of the subscriber.
   * @param string $langcode
   *   Langcode of the subscriber.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\TypedData\Exception\MissingDataException
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   * @throws \Drupal\de_notifications\Exception\NotificationsException
   */
  public function requestSubscriptionOverview(string $token, string $langcode): void;

}
