<?php

declare(strict_types=1);

namespace Drupal\de_notifications;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\de_notifications\Exception\NotificationsException;
use Firebase\JWT\ExpiredException;
use Firebase\JWT\JWT;
use Firebase\JWT\Key;

/**
 * Class NotificationsTokenService.
 *
 * Handles all the Token actions required for the Notifications.
 */
class NotificationsTokenService implements NotificationsTokenServiceInterface {

  /**
   * The constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The Config factory container.
   */
  public function __construct(protected readonly ConfigFactoryInterface $configFactory) {}

  /**
   * {@inheritDoc}
   */
  public function generateToken(string $uuid, int $ttl): string {
    $config = $this->configFactory->get('de_notifications.settings');

    $payload = [
      'sub' => $uuid,
    // Issued at time.
      'iat' => time(),
    // Expiry time.
      'exp' => time() + $ttl,
    ];

    return JWT::encode($payload, $config->get('secret_key'), 'HS256');
  }

  /**
   * {@inheritDoc}
   */
  public function validateToken(string $token): string {
    $config = $this->configFactory->get('de_notifications.settings');
    try {
      $decoded = JWT::decode($token, new Key($config->get('secret_key'), 'HS256'));
    }
    catch (ExpiredException) {
      // Token is expired.
      throw new NotificationsException('Token expired.', 410);
    }
    catch (\Exception) {
      // Token is invalid (tampered, or incorrect signature)
      throw new NotificationsException('Invalid token.', 401);
    }

    return $decoded->sub;

  }

}
