<?php

declare(strict_types=1);

namespace Drupal\de_notifications\Plugin\Field\FieldType;

use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\Core\Entity\RevisionableInterface;
use Drupal\Core\Entity\TranslatableInterface;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;

/**
 * Defines the 'notification_settings' field type.
 *
 * @FieldType(
 *   id = "notification_settings",
 *   label = @Translation("Notification settings"),
 *   description = @Translation("Add checkbox that allows users to subscribe to notifications for entity updates and add textfield to describe entity changes."),
 *   default_widget = "notification_settings",
 *   default_formatter = "notification_settings_default",
 *   cardinality = 1
 * )
 */
class NotificationSettingsItem extends FieldItemBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings() {
    return [
      'subscription_enabled_label' => new TranslatableMarkup('Visitors are allowed to subscribe'),
      'send_notification_label' => new TranslatableMarkup('Send notification'),
      'changes_label' => new TranslatableMarkup('Changes'),
      'changes_description' => new TranslatableMarkup('Changes to notify subscribers about'),
    ] + parent::defaultFieldSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty(): bool {
    return $this->subscription_enabled != 1 && $this->changes === NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state) {
    $element = [];

    $element['subscription_enabled_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subscription enabled "checkbox"-label'),
      '#default_value' => $this->getSetting('subscription_enabled_label'),
      '#required' => TRUE,
    ];
    $element['send_notification_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Send notification "checkbox"-label'),
      '#default_value' => $this->getSetting('send_notification_label'),
      '#required' => TRUE,
    ];
    $element['changes_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('"Changes"-label'),
      '#default_value' => $this->getSetting('changes_label'),
      '#required' => TRUE,
    ];
    $element['changes_description'] = [
      '#type' => 'textfield',
      '#title' => $this->t('"Changes"-description'),
      '#default_value' => $this->getSetting('changes_description'),
      '#required' => TRUE,
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition): array {
    $properties['subscription_enabled'] = DataDefinition::create('boolean')
      ->setLabel(t('Subscription enabled'));
    $properties['send_notification'] = DataDefinition::create('boolean')
      ->setLabel(t('Send notification'));
    $properties['changes'] = DataDefinition::create('string')
      ->setLabel(t('Changes'));

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition): array {
    $columns = [
      'subscription_enabled' => [
        'type' => 'int',
        'size' => 'tiny',
      ],
      'send_notification' => [
        'type' => 'int',
        'size' => 'tiny',
      ],
      'changes' => [
        'type' => 'text',
        'size' => 'big',
      ],
    ];

    return [
      'columns' => $columns,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function postSave($update) {
    if ($this->subscription_enabled !== 1 || $this->get('send_notification')->getValue() !== 1) {
      return FALSE;
    }

    $entity = $this->getEntity();
    // We can't do anything if we don't know the published state of the entity.
    if (!$entity instanceof EntityPublishedInterface) {
      return FALSE;
    }

    // Don't update users if entity is not default revision, i.e. drafts.
    if ($entity instanceof RevisionableInterface && !$entity->isDefaultRevision()) {
      return FALSE;
    }

    // Don't queue if the entity is published without changes or
    // the translation is not being inserted/updated.
    if ($entity->isPublished() &&
      ($this->isEmpty() ||
      ($entity instanceof TranslatableInterface && !$entity->hasTranslationChanges()))) {
      return FALSE;
    }

    $subscription_storage = \Drupal::service('entity_type.manager')->getStorage('de_notifications_subscription');
    /** @var \Drupal\de_notifications\NotificationsSubscriptionInterface[] $subscriptions */
    $subscriptions = $subscription_storage->loadByProperties([
      'entity' => [
        'target_type' => $entity->getEntityTypeId(),
        'target_id' => $entity->id(),
      ],
      'is_confirmed' => TRUE,
      'langcode' => $entity->language()->getId(),
    ]);

    foreach ($subscriptions as $subscription) {
      $queue = \Drupal::service('queue')->get('notify_subscribers');
      if ($entity->isPublished()) {
        $queue->createItem([
          'type' => 'update',
          'subscription' => $subscription,
          'changes' => $this->changes,
          'entity_title' => $entity->label(),
        ]);
      }
      else {
        // Entity is not published and not.
        $queue->createItem([
          'type' => 'archived',
          'subscription' => $subscription,
        ]);
      }
    }

    return FALSE;
  }

}
