<?php

declare(strict_types=1);

namespace Drupal\de_notifications\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\Validator\ConstraintViolationInterface;

/**
 * Defines the 'notification_settings' field widget.
 *
 * @FieldWidget(
 *   id = "notification_settings",
 *   label = @Translation("Notifications"),
 *   field_types = {"notification_settings"},
 * )
 */
class NotificationSettingsWidget extends WidgetBase {

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
    $entity = $items->getEntity();

    $element += [
      '#type' => 'details',
      '#group' => 'advanced',
    ];

    $element['subscription_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $items->getSetting('subscription_enabled_label'),
      '#default_value' => !empty($items[$delta]->subscription_enabled),
    ];

    $element['send_notification'] = [
      '#type' => 'checkbox',
      '#title' => $items->getSetting('send_notification_label'),
      '#default_value' => !$entity->isPublished() ? !empty($items[$delta]->send_notification) : 0,
      '#states' => [
        'visible' => [':input[name="' . $this->fieldDefinition->getName() . '[0][subscription_enabled]"]' => ['checked' => TRUE]],
      ],
    ];

    $conditional_value = [':input[name="' . $this->fieldDefinition->getName() . '[0][send_notification]"]' => ['checked' => TRUE]];

    $element['changes'] = [
      '#type' => 'textarea',
      '#title' => $items->getSetting('changes_label'),
      '#description' => $items->getSetting('changes_description'),
      '#default_value' => !$entity->isPublished() ? $items[$delta]->changes : '',
      '#states' => [
        // Show this textfield only if send_notification is true.
        'required' => $conditional_value,
        'visible' => $conditional_value,
      ],
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function errorElement(array $element, ConstraintViolationInterface $error, array $form, FormStateInterface $form_state): array|bool {
    $element = parent::errorElement($element, $error, $form, $form_state);
    if ($element === FALSE) {
      return FALSE;
    }
    $error_property = explode('.', $error->getPropertyPath())[1];
    return $element[$error_property] ?? FALSE;
  }

}
