<?php

declare(strict_types=1);

namespace Drupal\de_notifications\Plugin;

use Drupal\Component\Plugin\PluginInspectionInterface;
use Drupal\de_notifications\NotificationsSubscriberInterface;
use Drupal\de_notifications\NotificationsSubscriptionInterface;

/**
 * Defines an interface for notification type plugins.
 */
interface NotificationTypeInterface extends PluginInspectionInterface {

  /**
   * Get the notification type label.
   *
   * @return string
   *   Returns the plugin label.
   */
  public function label(): string;

  /**
   * Send subscribing user a confirmation notification with a link to confirm.
   *
   * @param \Drupal\de_notifications\NotificationsSubscriptionInterface $subscription
   *   The subscription.
   *
   * @throws \Drupal\Core\TypedData\Exception\MissingDataException
   */
  public function sendConfirmation(NotificationsSubscriptionInterface $subscription): void;

  /**
   * Send notification after successful confirmation.
   *
   * @param \Drupal\de_notifications\NotificationsSubscriptionInterface $subscription
   *   The subscription.
   *
   * @throws \Drupal\Core\TypedData\Exception\MissingDataException
   */
  public function sendSubscriptionConfirmed(NotificationsSubscriptionInterface $subscription): void;

  /**
   * Send a notification if a user is already subscribed.
   *
   * @param \Drupal\de_notifications\NotificationsSubscriptionInterface $subscription
   *   The subscription.
   *
   * @throws \Drupal\Core\TypedData\Exception\MissingDataException
   */
  public function sendAlreadySubscribed(NotificationsSubscriptionInterface $subscription): void;

  /**
   * Send a notification with an overview of all subscriptions to a subscriber.
   *
   * @param \Drupal\de_notifications\NotificationsSubscriberInterface $subscriber
   *   The subscriber.
   * @param string $langcode
   *   The langcode.
   *
   * @throws \Drupal\Core\TypedData\Exception\MissingDataException
   * @throws \Exception
   */
  public function sendSubscriptionOverview(NotificationsSubscriberInterface $subscriber, string $langcode): void;

  /**
   * Send a notification to notify subscribers about an entity update.
   *
   * @param \Drupal\de_notifications\NotificationsSubscriptionInterface $subscription
   *   The subscription.
   * @param string $entity_title
   *   Title of the entity that the notification is about.
   * @param string $changes
   *   Changes to the entity that the notification is about.
   *
   * @throws \Drupal\Core\TypedData\Exception\MissingDataException
   */
  public function sendEntityNotification(NotificationsSubscriptionInterface $subscription, string $entity_title, string $changes): void;

  /**
   * Send a notification if an entity is archived.
   *
   * @param \Drupal\de_notifications\NotificationsSubscriptionInterface $subscription
   *   The subscription.
   *
   * @throws \Drupal\Core\TypedData\Exception\MissingDataException
   */
  public function sendArchived(NotificationsSubscriptionInterface $subscription): void;

}
