<?php

namespace Drupal\de_notifications\Plugin\QueueWorker;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\de_notifications\Plugin\NotificationTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Process a queue of a subscription to notify.
 *
 * @QueueWorker(
 *   id = "notify_subscribers",
 *   title = @Translation("Notify Subscribers"),
 *   cron = {"time" = 600}
 * )
 */
class NotifySubscribers extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new class instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\de_notifications\Plugin\NotificationTypeManagerInterface $notificationTypeManager
   *   The notifications type plugin manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly NotificationTypeManagerInterface $notificationTypeManager,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('plugin.manager.notification_type'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data) {
    $subscription_storage = $this->entityTypeManager->getStorage('de_notifications_subscription');
    $config = $this->configFactory->get('de_notifications.settings');
    $plugin_id = $config->get('notification_type');
    if ($plugin_id && $this->notificationTypeManager->hasDefinition($plugin_id)) {
      /** @var \Drupal\de_notifications\Plugin\NotificationTypeInterface $notification_type */
      $notification_type = $this->notificationTypeManager->createInstance($plugin_id);
      if ($data['type'] === 'update') {
        $notification_type->sendEntityNotification(
          $data['subscription'],
          $data['entity_title'],
          $data['changes'],
        );
      }
      else {
        $subscription = $data['subscription'];
        $notification_type->sendArchived(
          $subscription,
        );

        $subscriber = $subscription->get('subscriber')->entity;
        $subscription->delete();

        $subscriptions = $subscription_storage->loadByProperties([
          'subscriber' => $subscriber->id(),
        ]);
        if (empty($subscriptions)) {
          $subscriber->delete();
        }
      }
    }
  }

}
