<?php

namespace Drupal\de_notifications\Plugin\Validation\Constraint;

use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;

/**
 * Validates the NotificationsPath constraint.
 */
class NotificationsPathConstraintValidator extends ConstraintValidator {

  /**
   * {@inheritdoc}
   */
  public function validate($value, Constraint $constraint) {
    if (empty($value)) {
      return;
    }

    // Path must start with a slash.
    if (!str_starts_with($value, '/')) {
      $this->context->addViolation($constraint->noLeadingSlash);
      return;
    }

    // Split path and query parameters.
    $parts = explode('?', $value);
    $path_part = $parts[0];
    $query_part = $parts[1] ?? '';

    // Check for trailing slash in path part.
    if (str_ends_with($path_part, '/')) {
      $this->context->addViolation($constraint->noTrailingSlash);
      return;
    }

    // Validate path part (everything before ?)
    if (preg_match('/[^a-zA-Z0-9\-_\/]/', $path_part)) {
      $this->context->addViolation($constraint->invalidPathChars);
    }

    // Validate query parameters if they exist.
    if (!empty($query_part)) {
      // Check if query parameters are properly formatted.
      if (!preg_match('/^[a-zA-Z0-9\-_=&]+$/', $query_part)) {
        $this->context->addViolation($constraint->invalidQueryChars);
        return;
      }

      // Validate each query parameter.
      $params = explode('&', $query_part);
      foreach ($params as $param) {
        if (!str_contains($param, '=')) {
          $this->context->addViolation($constraint->invalidQueryFormat);
          return;
        }

        [$key, $value] = explode('=', $param);
        if (empty($key) || empty($value)) {
          $this->context->addViolation($constraint->emptyKeyValue);
          return;
        }
      }
    }

    return $value;
  }

}
