<?php

namespace Drupal\de_notifications\Plugin\views\field;

use Drupal\Core\Database\Connection;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Field handler to expose confirmed subscribers count to views.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("notifications_subscriber_count")
 */
class NotificationsSubscriberCount extends FieldPluginBase implements ContainerFactoryPluginInterface {

  /**
   * Constructor to inject db connection.
   *
   * @param array $configuration
   *   The configuration.
   * @param string $plugin_id
   *   The plugin_id.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Database\Connection $db_connection
   *   The database connection.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger
   *   The logger factory.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected readonly Connection $db_connection,
    protected readonly LanguageManagerInterface $languageManager,
    protected readonly LoggerChannelFactoryInterface $logger,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('database'),
      $container->get('language_manager'),
      $container->get('logger.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function preRender(&$values): void {
    $entity_ids = [];
    $base_field = $this->view->storage->get('base_field');
    foreach ($values as $result) {
      if (isset($result->$base_field)) {
        $entity_ids[] = $result->$base_field;
      }
    }
    if (empty($entity_ids)) {
      return;
    }

    /** @var \Drupal\Core\Database\Query\Select $query */
    $query = $this->db_connection->select('de_notifications_subscription', 's');
    $query->addExpression('COUNT(entity__target_id)', 'notifications_subscribers_count');
    $query->addField('s', 'entity__target_id');
    $query->addField('s', 'langcode');
    // We only want subscriptions that are confirmed.
    $query->condition('s.entity__target_id', $entity_ids, 'IN');
    $query->condition('s.is_confirmed', 1);
    $query->groupBy('s.entity__target_id')->groupBy('s.langcode');

    try {
      $results = $query->execute()->fetchAll();
    }
    catch (\Exception $e) {
      $this->logger->get('notifications')->error($e->getMessage());
      return;
    }

    $counts = [];
    foreach ($results as $result) {
      $entity_id = $result->entity__target_id;
      $langcode = $result->langcode;
      if (!isset($counts[$entity_id])) {
        $counts[$entity_id] = [];
      }
      $counts[$entity_id][$langcode] = $result->notifications_subscribers_count;
    }

    foreach ($values as $result) {
      $base_id = $result->$base_field;
      $result->notifications_subscribers_count = $counts[$base_id] ?? [];
    }

  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    if (empty($values->notifications_subscribers_count)) {
      return '0';
    }
    // Set a default langcode value if it can't be retrieved from values.
    $langcode = $this->languageManager->getDefaultLanguage()->getId();
    $viewBaseTableLangCode = $this->view->storage->get('base_table') . '_langcode';
    if (isset($values->$viewBaseTableLangCode)) {
      $langcode = $values->$viewBaseTableLangCode;
    }
    if (isset($values->notifications_subscribers_count[$langcode])) {
      return $values->notifications_subscribers_count[$langcode];
    }
    return '0';
  }

  /**
   * {@inheritdoc}
   */
  public function query() {}

}
