<?php

declare(strict_types=1);

namespace Drupal\Tests\de_notifications\Kernel;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Url;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\KernelTests\KernelTestBase;
use Drupal\language\Entity\ConfigurableLanguage;
use Drupal\language\Plugin\LanguageNegotiation\LanguageNegotiationUrl;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Symfony\Component\HttpFoundation\Request;

/**
 * Tests the subscribe route.
 *
 * @group de_notifications
 */
class SubscribeTest extends KernelTestBase {

  use UserCreationTrait;

  /**
   * The kernel.
   *
   * @var \Symfony\Component\HttpKernel\HttpKernelInterface
   */
  protected $httpKernel;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'content_translation',
    'system',
    'field',
    'node',
    'user',
    'de_notifications',
    'de_notifications_test',
    'dynamic_entity_reference',
    'language',
    'text',
    'serialization',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('node');
    $this->installSchema('node', ['node_access']);
    $this->installEntitySchema('user');
    $this->installConfig(['de_notifications']);
    $this->installEntitySchema('de_notifications_subscriber');
    $this->installEntitySchema('de_notifications_subscription');

    $this->config('de_notifications.settings')
      ->set('notification_type', 'test_type')
      ->save();

    $this->installConfig(['language']);
    ConfigurableLanguage::createFromLangcode('nl')->save();

    $config = $this->config('language.types');
    $config->set('configurable', [
      LanguageInterface::TYPE_CONTENT,
    ]);
    $config->set('negotiation', [
      LanguageInterface::TYPE_CONTENT => [
        'enabled' => [LanguageNegotiationUrl::METHOD_ID => 0],
      ],
    ]);
    $config->save();
    $config = $this->config('language.negotiation');
    $config->set('url.prefixes', [
      'en' => 'en',
      'nl' => 'nl',
    ]);
    $config->save();
    $this->container->get('kernel')->rebuildContainer();

    NodeType::create(['type' => 'article'])->save();
    FieldStorageConfig::create([
      'field_name' => 'field_notifications',
      'type' => 'notification_settings',
      'entity_type' => 'node',
    ])->save();
    FieldConfig::create([
      'field_name' => 'field_notifications',
      'entity_type' => 'node',
      'bundle' => 'article',
      'label' => 'Notifications',
    ])->save();

    Node::create([
      'type' => 'article',
      'title' => 'Node 1',
      'field_notifications' => [
        'subscription_enabled' => TRUE,
      ],
    ])->save();

    Node::create([
      'type' => 'article',
      'title' => $this->randomString(),
      'field_notifications' => [
        'subscription_enabled' => FALSE,
      ],
    ])->save();

    $this->httpKernel = $this->container->get('http_kernel');
    $this->languageManager = $this->container->get('language_manager');
    $this->entityTypeManager = $this->container->get('entity_type.manager');

    $this->setUpCurrentUser([], [
      'create de_notifications_subscription',
      'create de_notifications_subscriber',
      'access content',
    ]);
  }

  /**
   * Data provider for ::testRequirementOnSubscribe.
   */
  public static function requirementsOnSubscribe(): array {
    return [
      [
        [
          'create de_notifications_subscription',
        ],
        [
          'email' => 'example@example.com',
          'entity_type' => 'node',
          'eid' => 1,
        ],
        [
          'X-Client-Ip' => '127.0.0.1',
        ],
        403,
        "The following permissions are required: 'create de_notifications_subscription' AND 'create de_notifications_subscriber'.",
      ],
      [
        [
          'create de_notifications_subscriber',
        ],
        [
          'email' => 'example@example.com',
          'entity_type' => 'node',
          'eid' => 1,
        ],
        [
          'X-Client-Ip' => '127.0.0.1',
        ],
        403,
        "The following permissions are required: 'create de_notifications_subscription' AND 'create de_notifications_subscriber'.",
      ],
      [
        [
          'create de_notifications_subscription',
          'create de_notifications_subscriber',
        ],
        [
          'entity_type' => 'node',
          'email' => 'test@example.com',
          'eid' => 1,
        ],
        ['X-Client-Ip' => '127.0.0.2'],
        200,
        'User was successfully subscribed.',
        NULL,
        [],
      ],
      [
        [
          'create de_notifications_subscription',
          'create de_notifications_subscriber',
          'access content',
        ],
        [
          'entity_type' => 'node',
          'email' => 'test@example.com',
          'eid' => 1,
        ],
        ['X-Client-Ip' => '127.0.0.2'],
        200,
        'User was successfully subscribed.',
        NULL,
        ['entityLabel' => 'Node 1'],
      ],
      [
        [
          'create de_notifications_subscription',
          'create de_notifications_subscriber',
        ],
        [
          'entity_type' => 'node',
          'eid' => 1,
        ],
        [
          'X-Client-Ip' => '127.0.0.1',
        ],
        400,
        'Parameter email is required.',
      ],
      [
        [
          'create de_notifications_subscription',
          'create de_notifications_subscriber',
        ],
        [
          'email' => 'example@example.com',
          'eid' => 1,
        ],
        [
          'X-Client-Ip' => '127.0.0.1',
        ],
        400,
        'Parameter entity_type is required.',
      ],
      [
        [
          'create de_notifications_subscription',
          'create de_notifications_subscriber',
        ],
        [
          'entity_type' => 'node',
          'email' => 'example@example.com',
        ],
        [
          'X-Client-Ip' => '127.0.0.1',
        ],
        400,
        'Parameter eid is required.',
      ],
      [
        [
          'create de_notifications_subscription',
          'create de_notifications_subscriber',
        ],
        [
          'entity_type' => 'node',
          'email' => 'example@example.com',
          'eid' => 1,
        ],
        [],
        400,
        'Header X-Client-Ip is required.',
      ],
      [
        [
          'create de_notifications_subscription',
          'create de_notifications_subscriber',
        ],
        [
          'entity_type' => 'non_existing_entity',
          'email' => 'example@example.com',
          'eid' => 1,
        ],
        [
          'X-Client-Ip' => '127.0.0.1',
        ],
        404,
        'Entity type non_existing_entity not found.',
      ],
      [
        [
          'create de_notifications_subscription',
          'create de_notifications_subscriber',
        ],
        [
          'entity_type' => 'node',
          'email' => 'example@example.com',
          'eid' => 3,
        ],
        [
          'X-Client-Ip' => '127.0.0.1',
        ],
        400,
        'The entity type node with id 3 does not exist.',
      ],
      [
        [
          'create de_notifications_subscription',
          'create de_notifications_subscriber',
        ],
        [
          'entity_type' => 'node',
          'email' => 'example@example.com',
          'eid' => 2,
        ],
        [
          'X-Client-Ip' => '127.0.0.1',
        ],
        403,
        'It is not allowed to subscribe to notifications on node with id 2.',
      ],
      [
        [
          'create de_notifications_subscription',
          'create de_notifications_subscriber',
        ],
        [
          'entity_type' => 'node',
          'email' => 'example@example.com',
          'eid' => 1,
        ],
        [
          'X-Client-Ip' => '127.0.0.1',
        ],
        400,
        'The langcode nl with entity id 1 does not exist.',
        'nl',
      ],
    ];
  }

  /**
   * Test requirements on subscribe request.
   *
   * @dataProvider requirementsOnSubscribe
   */
  public function testRequirementOnSubscribeRoute(array $permissions, array $params, array $headers, int $code, string $message, ?string $langcode = NULL, ?array $data = NULL): void {
    $this->setUpCurrentUser([], $permissions);
    $this->assertSubscribeResponse(
      $params,
      $headers,
      $code,
      $message,
      $langcode,
      $data
    );
  }

  /**
   * Test subscription.
   */
  public function testSubscription(): void {
    $this->assertSubscribeResponse(
      [
        'entity_type' => 'node',
        'email' => 'test@example.com',
        'eid' => 1,
      ],
      ['X-Client-Ip' => '127.0.0.2'],
      200,
      'User was successfully subscribed.',
      NULL,
      ['entityLabel' => 'Node 1']
    );

    $subscription_storage = $this->entityTypeManager->getStorage('de_notifications_subscription');
    /** @var \Drupal\de_notifications\NotificationsSubscriptionInterface $subscription */
    $subscription = current($subscription_storage->loadByProperties([
      'entity' => [
        'target_type' => 'node',
        'target_id' => 1,
      ],
    ]));
    $this->assertNotEmpty($subscription);
    $this->assertEquals(0, $subscription->get('is_confirmed')->value);
    $this->assertEquals(1, $subscription->get('entity')->entity->id());
    $this->assertEquals('en', $subscription->get('langcode')->value);
    $this->assertEquals('node', $subscription->get('entity')->entity->getEntityTypeId());
    $this->assertEquals('test@example.com', $subscription->get('subscriber')->entity->get('email')->value);
    $this->assertEquals('127.0.0.2', $subscription->get('subscriber')->entity->get('ip_address')->value);
  }

  /**
   * Test multilingual subscription.
   */
  public function testMultilingualSubscription(): void {
    $this->container->get('content_translation.manager')->setEnabled('node', 'article', TRUE);

    $node = Node::create([
      'type' => 'article',
      'title' => $this->randomString(),
      'field_notifications' => [
        'subscription_enabled' => FALSE,
      ],
    ]);
    $node->save();
    $translation = $node->addTranslation('nl', [
      'title' => $this->randomString(),
      'field_notifications' => [
        'subscription_enabled' => TRUE,
      ],
    ]);
    $translation->save();

    $params = [
      'entity_type' => $node->getEntityTypeId(),
      'email' => 'test@example.com',
      'eid' => $node->id(),
    ];
    $langcode = 'en';
    $headers = ['X-Client-Ip' => '127.0.0.1'];
    $this->assertSubscribeResponse(
      $params,
      $headers,
      403,
      "It is not allowed to subscribe to notifications on {$node->getEntityTypeId()} with id {$node->id()} and langcode {$langcode}.",
      $langcode,
    );

    $this->assertSubscribeResponse(
      $params,
      $headers,
      200,
      'User was successfully subscribed.',
      'nl',
      ['entityLabel' => $translation->label()]
    );

    $subscription_storage = $this->entityTypeManager->getStorage('de_notifications_subscription');
    /** @var \Drupal\de_notifications\NotificationsSubscriptionInterface $subscriptions */
    $subscriptions = $subscription_storage->loadByProperties([
      'entity' => [
        'target_type' => $node->getEntityTypeId(),
        'target_id' => $node->id(),
      ],
      'langcode' => 'nl',
    ]);
    $this->assertCount(1, $subscriptions);

    $node->set('field_notifications', [
      'subscription_enabled' => TRUE,
    ]);
    $node->save();

    // The correct translation of the entityLabel is given.
    $this->assertSubscribeResponse(
      $params,
      $headers,
      200,
      'User was successfully subscribed.',
      'en',
      ['entityLabel' => $node->label()]
    );
  }

  /**
   * Test already subscribed.
   */
  public function testAlreadySubscribed(): void {
    $this->assertSubscribeResponse(
      [
        'entity_type' => 'node',
        'email' => 'testAlreadySubscribed@example.com',
        'eid' => 1,
      ],
      ['X-Client-Ip' => '127.0.0.1'],
      200,
      'User was successfully subscribed.',
      NULL,
      ['entityLabel' => 'Node 1']
    );
    $this->assertMethodIsCalled('sendConfirmation');

    // Same subscription will send confirmation again.
    $this->assertSubscribeResponse(
      [
        'entity_type' => 'node',
        'email' => 'testAlreadySubscribed@example.com',
        'eid' => 1,
      ],
      ['X-Client-Ip' => '127.0.0.1'],
      200,
      'User was successfully subscribed.',
      NULL,
      ['entityLabel' => 'Node 1']
    );
    $this->assertMethodIsCalled('sendConfirmation');

    $subscription_storage = $this->entityTypeManager->getStorage('de_notifications_subscription');
    /** @var \Drupal\de_notifications\NotificationsSubscriptionInterface $subscription */
    $subscription = current($subscription_storage->loadByProperties([
      'entity' => [
        'target_type' => 'node',
        'target_id' => 1,
      ],
    ]));
    $subscription->set('is_confirmed', TRUE);
    $subscription->save();

    // Same subscription but this time the subscription is confirmed.
    $this->assertSubscribeResponse(
      [
        'entity_type' => 'node',
        'email' => 'testAlreadySubscribed@example.com',
        'eid' => 1,
      ],
      ['X-Client-Ip' => '127.0.0.1'],
      200,
      'User was successfully subscribed.',
      NULL,
      ['entityLabel' => 'Node 1']
    );
    $this->assertMethodIsCalled('sendAlreadySubscribed');
  }

  /**
   * Test subscriber exist.
   */
  public function testSubscriberExist(): void {
    // Initially there are no subscribers.
    $subscriber_storage = $this->entityTypeManager->getStorage('de_notifications_subscriber');
    $this->assertEmpty($subscriber_storage->loadMultiple());

    $this->assertSubscribeResponse(
      [
        'entity_type' => 'node',
        'email' => 'test@example.com',
        'eid' => 1,
      ],
      ['X-Client-Ip' => '127.0.0.2'],
      200,
      'User was successfully subscribed.',
      NULL,
      ['entityLabel' => 'Node 1']
    );

    // After the subscribing there should be only 1 subscriber.
    $this->assertCount(1, $subscriber_storage->loadMultiple());

    // Subscribe again.
    $this->assertSubscribeResponse(
      [
        'entity_type' => 'node',
        'email' => 'test@example.com',
        'eid' => 1,
      ],
      ['X-Client-Ip' => '127.0.0.2'],
      200,
      'User was successfully subscribed.',
      NULL,
      ['entityLabel' => 'Node 1']
    );

    // After the subscribing again there is still 1.
    $this->assertCount(1, $subscriber_storage->loadMultiple());
  }

  /**
   * Asserts the notification type method is called.
   *
   * @param string $method
   *   The method name.
   * @param bool $reset
   *   (optional) If TRUE, also clear the list of called methods for that type.
   */
  protected function assertMethodIsCalled(string $method, bool $reset = TRUE): void {
    $key = "de_notifications_test.test_type.method.{$method}";
    $value = \Drupal::state()->get($key, FALSE);
    if ($reset) {
      \Drupal::state()->delete($key);
    }
    $this->assertTrue($value);
  }

  /**
   * Assert subscribe response.
   *
   * @param array $params
   *   Request parameters.
   * @param array $headers
   *   Request headers.
   * @param int $expected_code
   *   The expected status code.
   * @param string $expected_message
   *   The expected message.
   * @param string|null $langcode
   *   Langcode to be used on the request.
   * @param array|null $expected_data
   *   The expected data.
   */
  protected function assertSubscribeResponse(array $params, array $headers, int $expected_code, string $expected_message, ?string $langcode = NULL, ?array $expected_data = NULL): void {
    $options = !empty($langcode) ? ['language' => $this->languageManager->getLanguage($langcode)] : [];
    $url = Url::fromRoute('de_notifications.subscribe', [], $options);
    $request = Request::create($url->toString(), 'POST', $params);
    $request->headers->add($headers);
    $response = $this->httpKernel->handle($request);
    $parsed_response = Json::decode((string) $response->getContent());
    $this->assertEquals($expected_code, $response->getStatusCode());
    $this->assertEquals($expected_message, $parsed_response['message']);
    $this->assertEquals($expected_data, array_key_exists('data', $parsed_response) ? $parsed_response['data'] : NULL);
  }

}
