<?php

namespace Drupal\dead_letter_queue\Queue;

use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Queue\DatabaseQueue;
use Drupal\Core\Queue\QueueWorkerManagerInterface;
use Drupal\dead_letter_queue\Exception\DiscardDeadLetterException;
use Drupal\dead_letter_queue\Exception\RestoreDeadLetterException;

/**
 * A dead letter database queue.
 */
class DeadLetterDatabaseQueue extends DatabaseQueue implements DeadLetterQueueInterface {

  /**
   * Constructs a DeadLetterDatabaseQueue object.
   *
   * @param mixed $name
   *   The name of the queue.
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   * @param \Drupal\Core\Queue\QueueWorkerManagerInterface $queueManager
   *   The queue manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger.
   */
  public function __construct(
    $name,
    Connection $connection,
    protected QueueWorkerManagerInterface $queueManager,
    protected ConfigFactoryInterface $configFactory,
    protected LoggerChannelInterface $logger,
  ) {
    parent::__construct($name, $connection);
  }

  /**
   * {@inheritdoc}
   */
  public function numberOfItems() {
    $maxTries = $this->getMaxTries();
    $query = 'SELECT COUNT(item_id) FROM {' . static::TABLE_NAME . '} WHERE name = :name AND tries < :max_tries';
    $args = [':name' => $this->name, ':max_tries' => $maxTries];

    try {
      return (int) $this->connection->query($query, $args)
        ->fetchField();
    }
    catch (\Exception $e) {
      $this->catchException($e);
      // If there is no table there cannot be any items.
      return 0;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function claimItem($lease_time = 30) {
    $maxTries = $this->getMaxTries();

    try {
      $queueWorker = $this->queueManager->createInstance($this->name);
    }
    catch (PluginNotFoundException $e) {
      $queueWorker = NULL;
    }

    // Claim an item by updating its expire fields. If claim is not successful
    // another thread may have claimed the item in the meantime. Therefore loop
    // until an item is successfully claimed or we are reasonably sure there
    // are no unclaimed items left.
    while (TRUE) {
      try {
        $query = 'SELECT data, created, item_id, tries FROM {' . static::TABLE_NAME . '} q WHERE expire = 0 AND name = :name AND tries < :max_tries ORDER BY created, item_id ASC';
        $args = [':name' => $this->name, ':max_tries' => $maxTries];
        $item = $this->connection->queryRange($query, 0, 1, $args)->fetchObject();
      }
      catch (\Exception $e) {
        $this->catchException($e);
      }

      // If the table does not exist there are no items currently available to
      // claim.
      if (empty($item)) {
        return FALSE;
      }

      // Increase the tries counter.
      $item->tries++;

      // Try to update the item. Only one thread can succeed in UPDATEing the
      // same row. We cannot rely on REQUEST_TIME because items might be
      // claimed by a single consumer which runs longer than 1 second. If we
      // continue to use REQUEST_TIME instead of the current time(), we steal
      // time from the lease, and will tend to reset items before the lease
      // should really expire.
      $update = $this->connection->update(static::TABLE_NAME)
        ->fields([
          'expire' => time() + $lease_time,
        ])
        ->expression('tries', 'tries+1')
        ->condition('item_id', $item->item_id)
        ->condition('expire', 0);

      // If there are affected rows, this update succeeded.
      if ($update->execute()) {
        $item->data = unserialize($item->data);

        if ($item->tries >= $maxTries) {
          $this->logger->error('Queue item @queueItemId from queue %queueName was moved to the dead letter queue after @tries tries.', [
            '@queueItemId' => $item->item_id,
            '%queueName' => $this->name,
            '@tries' => $maxTries,
          ]);

          if ($queueWorker instanceof DeadLetterQueueWorkerInterface) {
            try {
              $queueWorker->handleDeadLetter($item->data);
            }
            catch (DiscardDeadLetterException $e) {
              $this->deleteItem($item);
            }
            catch (RestoreDeadLetterException $e) {
              $this->resetItemTries($item->item_id);
            }
          }

          continue;
        }

        return $item;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function releaseItem($item) {
    try {
      $update = $this->connection->update(static::TABLE_NAME)
        ->fields([
          'expire' => 0,
        ])
        ->expression('tries', 'tries-1')
        ->condition('item_id', $item->item_id);
      return $update->execute();
    }
    catch (\Exception $e) {
      $this->catchException($e);
      // If the table doesn't exist we should consider the item released.
      return TRUE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function schemaDefinition() {
    $definition = parent::schemaDefinition();
    $definition['fields']['tries'] = [
      'type' => 'int',
      'not null' => TRUE,
      'default' => 0,
      'description' => 'Amount of times processing has been attempted.',
    ];

    return $definition;
  }

  /**
   * {@inheritdoc}
   */
  public function resetItemTries(int $itemId): void {
    $this->connection->update('queue')
      ->condition('item_id', $itemId)
      ->fields(['tries' => 0])
      ->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function getMaxTries(): int {
    $definition = $this->queueManager->getDefinition($this->name, FALSE);
    $config = $this->configFactory->get('dead_letter_queue.settings');

    foreach ($config->get('queues') ?? [] as $info) {
      if (!isset($info['queue_name'], $info['max_tries'])) {
        continue;
      }

      if ($info['queue_name'] === $this->name) {
        return $info['max_tries'];
      }
    }

    if (isset($definition['cron']['max_tries']) && is_int($definition['cron']['max_tries'])) {
      return $definition['cron']['max_tries'];
    }

    return 10;
  }

}
