<?php

namespace Drupal\debug_cacheability_headers_split\EventSubscriber;

use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Drupal\Core\Cache\CacheableResponseInterface;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Handles large debug cacheability headers by splitting them.
 */
class DebugCacheabilityHeadersSplitSubscriber implements EventSubscriberInterface {

  /**
   * The core header names for cacheability debugging.
   *
   * @var array
   */
  private static array $debugCacheabilityHeaderNames = [
    'X-Drupal-Cache-Tags', 'X-Drupal-Cache-Contexts',
  ];

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Whether cacheability headers for debugging purposes are enabled.
   *
   * @var bool
   */
  protected $debugCacheabilityHeaders = FALSE;

  /**
   * Constructs a DebugCacheabilityHeadersSplitSubscriber object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param bool $http_response_debug_cacheability_headers
   *   Whether cacheability headers for debugging purposes are enabled.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    $http_response_debug_cacheability_headers = FALSE,
  ) {
    $this->configFactory = $configFactory;
    $this->debugCacheabilityHeaders = $http_response_debug_cacheability_headers;
  }

  /**
   * Checks response headers for size limitation and applies split if needed.
   */
  public function onRespond(ResponseEvent $event) {
    if (!$event->isMainRequest()) {
      return;
    }

    // Don't act if debugging cacheability headers is not enabled.
    if (!$this->debugCacheabilityHeaders) {
      return;
    }

    $response = $event->getResponse();

    // Don't act if the current response isn't cacheable.
    if (!$response instanceof CacheableResponseInterface) {
      return;
    }

    $config = $this->configFactory->get('debug_cacheability_headers_split.settings');

    foreach (self::$debugCacheabilityHeaderNames as $name) {
      $value = $response->headers->get($name);
      // If the value exceeds the limit, proceed to splitting.
      // Note that, since we are splitting words (each cache tag/context literal
      // is a word) and words are not cut of, it is theoretically possible
      // that we end up with a header that exceeds the limit if there is a word
      // that also exceeds this limit. In practice this will not happen, as the
      // minimum header limit is 1024 bytes and there is no tag or
      // context literal that exceeds this.
      if (!empty($value) && strlen($value) > $config->get('header_size_limit')) {
        foreach (explode("\n", wordwrap($value, $config->get('header_chunk_size'))) as $delta => $row) {
          $response->headers->set($name . ($delta > 0 ? "-{$delta}" : ''), $row);
        }
      }
    }

  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    // Should run after FinishResponseSubscriber.
    return [
      KernelEvents::RESPONSE => ['onRespond', -500],
    ];
  }

}
