<?php

namespace Drupal\debug_cacheability_headers_split\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure settings for debugging cacheability headers.
 */
class DebugCacheabilityHeadersSplitSettingsForm extends ConfigFormBase {

  // 1KB
  private const MIN_HEADER_SIZE = 1024;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'debug_cacheability_headers_split_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['debug_cacheability_headers_split.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('debug_cacheability_headers_split.settings');

    $form['header_size_limit'] = [
      '#type' => 'number',
      '#title' => $this->t('Header size limit (bytes)'),
      '#description' => $this->t('Set the maximum header size limit above which debug cacheability headers should be split (minimum: 1024).'),
      '#default_value' => $config->get('header_size_limit'),
      '#min' => self::MIN_HEADER_SIZE,
      '#required' => TRUE,
    ];

    $form['header_chunk_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum header chunk size (bytes)'),
      '#description' => $this->t('Set the max size for each header chunk when debug cacheability headers are split (minimum: 512).'),
      '#default_value' => $config->get('header_chunk_size'),
      '#min' => self::MIN_HEADER_SIZE / 2,
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
    $header_size_limit = $form_state->getValue('header_size_limit');
    if ($header_size_limit < self::MIN_HEADER_SIZE) {
      $form_state->setErrorByName('header_size_limit', $this->t('The header limit must be at least 1024 bytes.'));
    }
    $header_chunk_size = $form_state->getValue('header_chunk_size');
    if ($header_chunk_size < (self::MIN_HEADER_SIZE / 2)) {
      $form_state->setErrorByName('header_chunk_size', $this->t('The max header chunk size must be at least 512 bytes.'));
    }
    if ($header_size_limit <= $header_chunk_size) {
      $form_state->setErrorByName('header_size_limit', $this->t('The header limit must be greater than the max chunk size.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('debug_cacheability_headers_split.settings')
      ->set('header_size_limit', $form_state->getValue('header_size_limit'))
      ->set('header_chunk_size', $form_state->getValue('header_chunk_size'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
