<?php

namespace Drupal\Tests\debug_cacheability_headers_split\Unit\EventSubscriber;

use Drupal\debug_cacheability_headers_split\EventSubscriber\DebugCacheabilityHeadersSplitSubscriber;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\HttpKernelInterface;
use Drupal\Core\Cache\CacheableResponse;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * @coversDefaultClass \Drupal\debug_cacheability_headers_split\EventSubscriber\DebugCacheabilityHeadersSplitSubscriber
 * @group debug_cacheability_headers_split
 */
class DebugCacheabilityHeadersSplitSubscriberTest extends UnitTestCase {

  /**
   * The mock Kernel.
   *
   * @var \Symfony\Component\HttpKernel\HttpKernelInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $kernel;

  /**
   * The mock config factory.
   *
   * @var \Symfony\Component\HttpKernel\HttpKernelInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->kernel = $this->createMock(HttpKernelInterface::class);

    $configMock = $this->createMock(Config::class);
    $configMock->method('get')
      ->willReturnCallback(function ($key) {
        switch ($key) {
          case 'header_size_limit':
            return 32;

          case 'header_chunk_size':
            return 16;

          default:
            return 0;
        }
      });

    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->configFactory->expects($this->any())
      ->method('get')
      ->with('debug_cacheability_headers_split.settings')
      ->willReturn($configMock);
  }

  /**
   * Tests that debug cacheability headers are not processed if within limit.
   *
   * @covers ::onRespond
   */
  public function testDebugCacheabilityHeadersWithinLimit() {

    $subscriber = new DebugCacheabilityHeadersSplitSubscriber($this->configFactory, TRUE);

    $request = $this->createMock(Request::class);

    $response = new CacheableResponse('OK', '200', [
      'X-Drupal-Cache-Tags' => 'node:1 node:2',
      'X-Drupal-Cache-Contexts' => 'url',
    ]);

    $event = new ResponseEvent($this->kernel, $request, HttpKernelInterface::MAIN_REQUEST, $response);

    $subscriber->onRespond($event);

    $this->assertEquals('node:1 node:2', $response->headers->get('X-Drupal-Cache-Tags'));
    $this->assertEquals('url', $response->headers->get('X-Drupal-Cache-Contexts'));
    $this->assertNull($response->headers->get('X-Drupal-Cache-Tags-1'));
    $this->assertNull($response->headers->get('X-Drupal-Cache-Tags-2'));
    $this->assertNull($response->headers->get('X-Drupal-Cache-Contexts-1'));
  }

  /**
   * Tests that debug cacheability headers are split if limit is exceeded.
   *
   * @covers ::onRespond
   */
  public function testDebugCacheabilityHeadersSplit() {

    $subscriber = new DebugCacheabilityHeadersSplitSubscriber($this->configFactory, TRUE);

    $request = $this->createMock(Request::class);

    $response = new CacheableResponse('OK', '200', [
      'X-Drupal-Cache-Tags' => 'node:1 node:2 node:3 node:4 node:5 node:6 node:7 config:paragraphs.paragraphs_type.foobar',
      'X-Drupal-Cache-Contexts' => 'url route',
    ]);

    $event = new ResponseEvent($this->kernel, $request, HttpKernelInterface::MAIN_REQUEST, $response);

    $subscriber->onRespond($event);

    // Assert that this header is within limit and not split.
    $this->assertEquals('url route', $response->headers->get('X-Drupal-Cache-Contexts'));

    // Assert splitting the cache tags into multiple headers.
    $this->assertEquals('node:1 node:2', $response->headers->get('X-Drupal-Cache-Tags'));
    $this->assertEquals('node:3 node:4', $response->headers->get('X-Drupal-Cache-Tags-1'));
    $this->assertEquals('node:5 node:6', $response->headers->get('X-Drupal-Cache-Tags-2'));
    $this->assertEquals('node:7', $response->headers->get('X-Drupal-Cache-Tags-3'));
    // The actual header size here is bigger than the configured chunk size,
    // because whole words are not cut, even if header limit is exceeded.
    $this->assertEquals('config:paragraphs.paragraphs_type.foobar', $response->headers->get('X-Drupal-Cache-Tags-4'));
    $this->assertNull($response->headers->get('X-Drupal-Cache-Tags-5'));
  }

}
