<?php

namespace Drupal\decoupled_kit_block\Resolver;

use Drupal\Core\Breadcrumb\Breadcrumb;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Url;
use Drupal\decoupled_kit\DecoupledKit;
use Drupal\views\Views;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Alter breadcrumb.
 */
class BreadcrumbAlter implements ContainerInjectionInterface {

  /**
   * Constructs a BreadcrumbAlter object.
   */
  public function __construct(
    private readonly DecoupledKit $decoupledKit,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('decoupled_kit'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * Implements hook_system_breadcrumb_alter().
   */
  public function alterBreadcrumb(Breadcrumb &$breadcrumb, RouteMatchInterface $route_match, array $context) {
    $need_alter = FALSE;
    foreach ($breadcrumb->getLinks() as $link) {
      if ($link->getText() === 'Jsonapi') {
        $need_alter = TRUE;
        break;
      }
    }

    if (!$need_alter) {
      return;
    }

    $path = Url::fromRouteMatch($route_match)->toString();
    $path_elements = explode('/', trim($path, '/'));
    while (count($path_elements) > 1) {
      array_pop($path_elements);
      $path = $this->decoupledKit->canonicalPath(implode('/', $path_elements));
      $entity = $this->decoupledKit->getEntityFromPath($path);
      if (!$entity) {
        break;
      }

      $title = $this->getTitle($path, $entity);
      if ($title) {
        $breadcrumb->addLink(Link::fromTextAndUrl($title, $this->getUrl($path)));
      }
    }
  }

  /**
   * Get title for path and entity.
   *
   * @param string $path
   *   The path value.
   * @param object $entity
   *   The entity object.
   *
   * @return string|null
   *   Title value or NULL.
   */
  protected function getTitle($path, $entity) {
    try {
      $routeName = $this->getUrl($path)->getRouteName();
    }
    catch (\Exception $e) {
      return NULL;
    }

    $title = NULL;
    $routeParts = explode('.', $routeName);
    switch ($routeParts[0]) {
      case 'entity':
        $title = $this->getEntityTitle($entity);
        break;

      case 'view':
        $viewName = $routeParts[1];
        $displayId = $routeParts[2];
        $view = Views::getView($viewName);
        if (!$view || !$view->access($displayId)) {
          return NULL;
        }
        $view->setDisplay($displayId);
        $title = $view->getTitle();
        break;
    }

    return $title;
  }

  /**
   * Get entity title.
   *
   * @param object $entity
   *   Entity object.
   *
   * @return string
   *   Title value.
   */
  protected function getEntityTitle($entity) {
    $entity_type = $entity->getEntityTypeId();
    $label_key = $this->entityTypeManager->getDefinition($entity_type)->getKey('label');
    $title = $entity->get($label_key)->getString();

    return $title;
  }

  /**
   * Get URL from path.
   *
   * @param string $path
   *   The path value.
   *
   * @return \Drupal\Core\Url
   *   The URL object.
   */
  protected function getUrl($path) {
    return Url::fromUri("internal:" . $path);
  }

}
