<?php

namespace Drupal\decoupled_kit_block\Resource;

use Drupal\block\BlockRepositoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Path\PathMatcherInterface;
use Drupal\Core\Theme\ThemeManagerInterface;
use Drupal\Core\Breadcrumb\BreadcrumbManager;
use Drupal\decoupled_kit\DecoupledKit;
use Drupal\jsonapi\ResourceResponse;
use Drupal\jsonapi_resources\Resource\EntityResourceBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Processes a request for a collection of blocks.
 *
 * @internal
 */
final class Blocks extends EntityResourceBase implements ContainerInjectionInterface {

  /**
   * Constructs a Blocks object.
   */
  public function __construct(
    private readonly DecoupledKit $decoupledKit,
    private readonly ThemeManagerInterface $themeManager,
    private readonly PathMatcherInterface $pathMatcher,
    private readonly BreadcrumbManager $breadcrumbManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('decoupled_kit'),
      $container->get('theme.manager'),
      $container->get('path.matcher'),
      $container->get('breadcrumb')
    );
  }

  /**
   * Process the resource request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return \Drupal\jsonapi\ResourceResponse
   *   The response.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function process(Request $request): ResourceResponse {
    $path = $this->decoupledKit->checkPath($request);
    if ($path == '/') {
      $path = '<front>';
    }

    if ($request->query->has('current_theme')) {
      $theme = trim($request->query->get('current_theme'));
    }
    else {
      $theme = $this->themeManager->getActiveTheme()->getName();
    }

    $regions = system_region_list($theme, BlockRepositoryInterface::REGIONS_VISIBLE);
    if ($request->query->has('selected_regions')) {
      $selected_regions = trim($request->query->get('selected_regions'));
      $region_keys = array_map('trim', explode(',', $selected_regions));
      $regions = array_intersect_key($regions, array_flip($region_keys));
    }

    $blocks = $this->getBlocks($path, $theme, $regions);

    // Clean visibility setting.
    $clean_blocks = [];
    foreach ($blocks as $block) {
      $cloned = clone $block;
      $cloned->set('visibility', []);
      $clean_blocks[] = $cloned;
    }

    $data = $this->createCollectionDataFromEntities($clean_blocks, TRUE);
    return $this->createJsonapiResponse($data, $request);
  }

  /**
   * Block objects list.
   *
   * @param string $path
   *   Path to the checking page.
   * @param string $theme
   *   Theme name.
   * @param array $regions
   *   Regions list.
   *
   * @return array
   *   Blocks array.
   */
  protected function getBlocks(string $path, string $theme, array $regions): array {
    $blocks = [];
    $blocksManager = $this->entityTypeManager->getStorage('block');
    foreach (array_keys($regions) as $region) {
      $region_blocks_id = $blocksManager
        ->getQuery()
        ->accessCheck(TRUE)
        ->condition('theme', $theme)
        ->condition('region', $region)
        ->condition('status', 1)
        ->sort('weight')
        ->execute();

      if (!empty($region_blocks_id)) {
        $region_blocks = $blocksManager->loadMultiple($region_blocks_id);
        $region_blocks = (array) $region_blocks;
        $blocks = array_merge($blocks, $region_blocks);
      }
    }

    $blocks = array_filter($blocks, function ($block) use ($path) {
      return $this->blockVisibleForPage($block, $path);
    });

    $blocks = array_map(function ($block) use ($path) {
      $settings = $block->get('settings');

      // Keep system breadcrumb block.
      if ($settings['id'] === 'system_breadcrumb_block') {
        $breadcrumb = $this->getBreadcrumb($path);
        if ($breadcrumb) {
          $block->set('settings', [
            ...$settings,
            'breadcrumb' => $breadcrumb,
          ]);
          $block->save();
        }
      }
      return $block;
    }, $blocks);

    return $blocks;
  }

  /**
   * Visibility check.
   *
   * @param object $block
   *   Block object.
   * @param string $input_path
   *   Path to the checking page.
   *
   * @return bool
   *   Is the block visibility?
   */
  protected function blockVisibleForPage($block, $input_path): bool {
    $visibility = $block->getVisibility();
    if (empty($visibility)) {
      return TRUE;
    }

    // Request path visibility.
    $requestPathVisibility = TRUE;
    $requestPath = $visibility['request_path'] ?? NULL;
    if (!empty($requestPath)) {
      $negate = !empty($requestPath['negate']);
      if (empty($requestPath['pages'])) {
        $requestPathVisibility = !$negate;
      }
      else {
        $match = $input_path === $requestPath['pages'] || $this->pathMatcher->matchPath($input_path, $requestPath['pages']);
        $requestPathVisibility = $negate ? !$match : $match;
      }
    }

    // User roles visibility.
    $userRolesVisibility = TRUE;
    $userRoles = $visibility['user_role'] ?? NULL;
    if (!empty($userRoles)) {
      $negate = !empty($userRoles['negate']);
      if (empty($userRoles['roles'])) {
        $userRolesVisibility = !$negate;
      }
      else {
        $userRolesVisibility = !empty(array_intersect($userRoles['roles'], $this->currentUser()->getRoles()));
        if ($negate) {
          $userRolesVisibility = !$userRolesVisibility;
        }
      }
    }

    // Node type visibility.
    $nodeTypeVisibility = TRUE;
    $nodeType = $visibility['node_type'] ?? NULL;
    if (!$nodeType) {
      $nodeType = $visibility['entity_bundle:node'] ?? NULL;
    }
    if (!empty($nodeType)) {
      $negate = !empty($nodeType['negate']);
      if (empty($nodeType['bundles'])) {
        $nodeTypeVisibility = !$negate;
      }
      else {
        $entity = $this->decoupledKit->getEntityFromPath($input_path);
        if (!empty($entity) && $entity->getEntityTypeId() == 'node') {
          $nodeTypeVisibility = in_array($entity->getType(), $nodeType['bundles']);
          if ($negate) {
            $nodeTypeVisibility = !$nodeTypeVisibility;
          }
        }
      }
    }

    return $requestPathVisibility && $userRolesVisibility && $nodeTypeVisibility;
  }

  /**
   * Get breadcrumb.
   *
   * @param string $path
   *   Current path.
   *
   * @return array|bool
   *   Breadcrumb data.
   */
  protected function getBreadcrumb($path) {
    $route_match = $this->decoupledKit->getRouteMatchFromPath($path);
    if ($route_match) {
      $breadcrumb = $this->breadcrumbManager->build($route_match);

      $links = $breadcrumb->getLinks();

      // Remove the failed Jsonapi breadcrumb.
      $fail_index = NULL;
      foreach ($links as $index => $link) {
        if ($link->getText() === 'Jsonapi') {
          $fail_index = $index;
          break;
        }
      }

      if (!is_null($fail_index)) {
        $cloned_links = $links;
        $links = [];
        foreach ($cloned_links as $index => $link) {
          if ($index > 0 && $index > $fail_index) {
            $links[] = $link;
          }
        }
        $links = array_reverse($links);
        array_unshift($links, reset($cloned_links));
      }

      $links = array_map(function ($link) {
        return [
          'text' => $link->getText(),
          'url' => $link->getUrl()->toString(),
        ];
      }, $links);

      return $links;
    }
  }

}
