<?php

namespace Drupal\decoupled_kit_redirect\Resource;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\decoupled_kit\DecoupledKit;
use Drupal\jsonapi\ResourceResponse;
use Drupal\jsonapi\JsonApiResource\JsonApiDocumentTopLevel;
use Drupal\jsonapi\JsonApiResource\LinkCollection;
use Drupal\jsonapi\JsonApiResource\NullIncludedData;
use Drupal\jsonapi\JsonApiResource\ResourceObjectData;
use Drupal\jsonapi_resources\Resource\EntityResourceBase;
use Drupal\path_alias\AliasManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Processes a request for a collection of featured nodes.
 *
 * @internal
 */
final class Redirect extends EntityResourceBase implements ContainerInjectionInterface {

  /**
   * Constructs a Blocks object.
   */
  public function __construct(
    private readonly DecoupledKit $decoupledKit,
    private readonly AliasManager $aliasManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('decoupled_kit'),
      $container->get('path_alias.manager'),
    );
  }

  /**
   * Process the resource request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return \Drupal\jsonapi\ResourceResponse
   *   The response.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function process(Request $request): ResourceResponse {
    $path = $this->decoupledKit->checkPath($request, FALSE);
    $inner_path = $this->aliasManager->getPathByAlias($path);

    $entities = $this->entityTypeManager
      ->getStorage('redirect')
      ->loadByProperties([
        'redirect_source.path' => ltrim($inner_path, '/'),
      ]);

    if (empty($entities)) {
      $document = new JsonApiDocumentTopLevel(new ResourceObjectData([]), new NullIncludedData(), new LinkCollection([]));
      $response = $this->createJsonapiResponse($document->getData(), $request);
    }
    else {
      $entity = reset($entities);
      $data = $this->createIndividualDataFromEntity($entity);

      // Add meta.
      $redirect_to = $entity->get('redirect_redirect')->getValue();
      $uri = current($redirect_to)['uri'];
      $system_path = preg_replace('/^[^:]+:/', '/', $uri);
      $alias = $this->aliasManager->getAliasByPath($system_path);

      $meta = [
        'alias' => $alias,
      ];

      $response = $this->createJsonapiResponse($data, $request, 200, [], NULL, $meta);
    }

    $cacheability = new CacheableMetadata();
    $cacheability->addCacheContexts(['url.query_args:current_page']);
    $response->addCacheableDependency($cacheability);

    return $response;
  }

}
