<?php

declare(strict_types=1);

namespace Drupal\decoupled_kit;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\TranslatableInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Path\PathValidator;
use Drupal\Core\Routing\RouteMatch;
use Drupal\Core\Routing\RouteProvider;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * The Decoupled Kit service.
 */
final class DecoupledKit implements DecoupledKitInterface {

  /**
   * Constructs a DecoupledKit object.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly LanguageManagerInterface $languageManager,
    private readonly PathValidator $pathValidator,
    private readonly RouteProvider $routeProvider,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function canonicalPath($path, $url_path_only = TRUE): string {
    $path = mb_strtolower(trim($path));
    if ($url_path_only) {
      $path = parse_url($path, PHP_URL_PATH);
    }
    return sprintf('/%s', ltrim($path, '/'));
  }

  /**
   * {@inheritdoc}
   */
  public function checkPath(Request $request, $needCanonicalUrl = TRUE): string {
    $key_parameter = 'current_path';
    $path = $request->query->get($key_parameter);
    if (empty($path)) {
      throw new NotFoundHttpException("Unable to work with empty \"$key_parameter\". Please send a \"$key_parameter\" query string parameter with your request.");
    }

    if ($needCanonicalUrl) {
      $path = $this->canonicalPath($path);
    }
    return $path;
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityFromPath($path, $checkAllow = TRUE) {
    if (empty($path)) {
      return NULL;
    }

    try {
      $params = Url::fromUri("internal:" . $path)->getRouteParameters();
    }
    catch (\Exception $e) {
      return NULL;
    }
    $entityType = key($params);
    if (empty($entityType)) {
      return NULL;
    }

    $id = $params[$entityType];

    // View support.
    if ($entityType === 'view_id') {
      $entityType = 'view';
    }

    $entity = $this->entityTypeManager
      ->getStorage($entityType)
      ->load($id);

    if ($entity && $entityType === 'view') {
      if (isset($params['display_id'])) {
        $entity = $entity->getDisplay($params['display_id']);
      }
    }

    // Check translation.
    if ($entity instanceof TranslatableInterface) {
      $language = $this->languageManager->getCurrentLanguage()->getId();
      if ($entity->hasTranslation($language)) {
        $entity = $entity->getTranslation($language);
      }
    }

    return $entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getRouteMatchFromPath($path) {
    $url = $this->pathValidator->getUrlIfValid($path);
    if (!$url) {
      return NULL;
    }

    $route = $this->routeProvider->getRouteByName($url->getRouteName());
    $route_match = new RouteMatch(
      $url->getRouteName(),
      $route,
      $url->getRouteParameters(),
      $url->getRouteParameters()
    );

    return $route_match;
  }

}
