<?php

declare(strict_types=1);

namespace Drupal\decoupled_kit\Form;

use Drupal\block\BlockRepositoryInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a Decoupled Kit form.
 */
final class DashboardForm extends FormBase {

  /**
   * Drupal\Core\Extension\ModuleHandlerInterface definition.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Decoupled Kit service.
   *
   * @var \Drupal\decoupled_kit\DecoupledKitInterface
   */
  protected $decoupledKit;

  /**
   * Theme manager.
   *
   * @var \Drupal\Core\Theme\ThemeManagerInterface
   */
  protected $themeManager;

  /**
   * Theme handler.
   *
   * @var \Drupal\Core\Extension\ThemeHandlerInterface
   */
  protected $themeHandler;

  /**
   * Current page path for dashboard.
   *
   * @var string
   */
  protected $currentPath;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->moduleHandler = $container->get('module_handler');
    $instance->decoupledKit = $container->get('decoupled_kit');
    $instance->themeManager = $container->get('theme.manager');
    $instance->themeHandler = $container->get('theme_handler');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'decoupled_kit_dashboard';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $request = $this->getRequest();

    $current_path = $request->query->get('current_path') ??
    $this->config('decoupled_kit.config')->get('current_path') ??
    '/';

    $form['current_path'] = [
      '#title' => $this->t('Current path for dashboard links'),
      '#type' => 'textfield',
      '#required' => TRUE,
      '#default_value' => $this->decoupledKit->canonicalPath($current_path),
      '#weight' => -2,
    ];

    if ($this->moduleHandler->moduleExists('decoupled_kit_block')) {
      $form['blocks_options'] = [
        '#type' => 'details',
        '#title' => $this->t('Blocks options'),
        '#open' => FALSE,
      ];

      $themes = [];
      foreach ($this->themeHandler->listInfo() as $machine_name => $theme_info) {
        $themes[$machine_name] = $theme_info->info['name'];
      }

      if ($form_state->hasValue('current_theme')) {
        $current_theme = $form_state->getValue('current_theme');
      }
      elseif ($request->query->has('current_theme')) {
        $current_theme = trim($request->query->get('current_theme'));
      }
      else {
        $current_theme = $this->themeManager->getActiveTheme()->getName();
      }

      $form['blocks_options']['current_theme'] = [
        '#type' => 'select',
        '#title' => $this->t('Theme'),
        '#description' => $this->t('Select a theme'),
        '#options' => $themes,
        '#default_value' => $current_theme,
        '#ajax' => [
          'callback' => [$this, 'updateRegionsCallback'],
          'wrapper' => 'selected-regions-wrapper',
          'event' => 'change',
          'progress' => [
            'type' => 'throbber',
            'message' => $this->t('Loading regions...'),
          ],
        ],
      ];

      if ($form_state->isRebuilding()) {
        $selected_regions = [];
        $user_input = $form_state->getUserInput();
        if (isset($user_input['selected_regions'])) {
          $user_input['selected_regions'] = [];
          $form_state->setUserInput($user_input);
        }
      }
      elseif ($form_state->hasValue('selected_regions')) {
        $selected_regions = array_filter($form_state->getValue('selected_regions'));
      }
      elseif ($request->query->has('selected_regions')) {
        $selected_regions = trim($request->query->get('selected_regions'));
        $selected_regions = array_map('trim', explode(',', $selected_regions));
      }
      else {
        $selected_regions = [];
      }

      $regions = system_region_list($current_theme, BlockRepositoryInterface::REGIONS_VISIBLE);

      $form['blocks_options']['selected_regions_wrapper'] = [
        '#type' => 'container',
        '#attributes' => ['id' => 'selected-regions-wrapper'],
      ];

      $form['blocks_options']['selected_regions_wrapper']['selected_regions'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Regions'),
        '#description' => $this->t("Choose theme's regions"),
        '#options' => $regions,
        '#default_value' => $selected_regions,
      ];
    }

    $form['actions'] = [
      '#type' => 'actions',
      '#weight' => 10,
    ];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Set parameters'),
    ];

    $form['table'] = [
      '#title' => $this->t('Table of functions'),
      '#type' => 'table',
      '#rows' => $this->getTableRows($current_path, $current_theme, $selected_regions),
      '#weight' => 99,
    ];

    return $form;
  }

  /**
   * AJAX callback to update the regions list.
   */
  public function updateRegionsCallback(array &$form, FormStateInterface $form_state) {
    $current_theme = $form_state->getValue('current_theme');
    $regions = system_region_list($current_theme, BlockRepositoryInterface::REGIONS_VISIBLE);

    $form['blocks_options']['selected_regions_wrapper']['selected_regions']['#options'] = $regions;
    $form['blocks_options']['selected_regions_wrapper']['selected_regions']['#default_value'] = [];

    return $form['blocks_options']['selected_regions_wrapper'];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $current_path = $this->decoupledKit->canonicalPath($form_state->getValue('current_path'));
    $this->configFactory()
      ->getEditable('decoupled_kit.config')
      ->set('current_path', $current_path)
      ->save();

    $query = ['current_path' => $current_path];

    $current_theme = $form_state->getValue('current_theme');
    if (!empty($current_theme)) {
      $query['current_theme'] = $current_theme;
    }

    $selected_regions = array_filter($form_state->getValue('selected_regions'));
    if (!empty($selected_regions)) {
      $query['selected_regions'] = implode(',', $selected_regions);
    }

    $form_state->setRedirect('decoupled_kit.dashboard', [], ['query' => $query]);
  }

  /**
   * Get table rows.
   *
   * @param string $current_path
   *   Current path.
   * @param string $current_theme
   *   Current theme.
   * @param array $selected_regions
   *   List of selected theme's regions.
   *
   * @return array
   *   The table of links.
   */
  protected function getTableRows($current_path, $current_theme, array $selected_regions): array {
    $functions = [
      [$this->t('Router'), 'decoupled_kit.router'],
    ];

    $modules = [
      'decoupled_kit_block' => [$this->t('Blocks'), 'decoupled_kit.block'],
      'decoupled_kit_redirect' => [$this->t('Redirect'), 'decoupled_kit.redirect'],
    ];

    foreach ($modules as $module => $data) {
      if ($this->moduleHandler->moduleExists($module)) {
        $functions[$module] = $data;
      }
    }

    $rows = [];
    foreach ($functions as $module => $function) {
      [$title, $route] = $function;

      $query = ['current_path' => $current_path];

      if ($module === 'decoupled_kit_block') {
        $query['current_theme'] = $current_theme;
        if (!empty($selected_regions)) {
          $query['selected_regions'] = implode(',', $selected_regions);
        }
      }

      $url = Url::fromRoute($route, [], [
        'query' => $query,
        'attributes' => ['target' => '_blank'],
      ]);
      $rows[] = [Link::fromTextAndUrl($title, $url)];
    }
    return $rows;
  }

}
