<?php

namespace Drupal\decoupled_kit\Resource;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\decoupled_kit\DecoupledKit;
use Drupal\jsonapi\JsonApiResource\ResourceObjectData;
use Drupal\jsonapi\ResourceResponse;
use Drupal\jsonapi_resources\Resource\EntityResourceBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Routing\Route;

/**
 * Processes a request for a collection of featured nodes.
 *
 * @internal
 */
final class Router extends EntityResourceBase implements ContainerInjectionInterface {

  /**
   * Routed entity.
   *
   * @var object
   */
  private $entity = NULL;

  /**
   * Constructs a Router object.
   */
  public function __construct(
    private readonly DecoupledKit $decoupledKit,
    private readonly RequestStack $requestStack,
  ) {
    $request = $this->requestStack->getCurrentRequest();
    $path = $this->decoupledKit->checkPath($request);
    $this->entity = $this->decoupledKit->getEntityFromPath($path);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('decoupled_kit'),
      $container->get('request_stack'),
    );
  }

  /**
   * Process the resource request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return \Drupal\jsonapi\ResourceResponse
   *   The response.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function process(Request $request): ResourceResponse {
    if (!$this->entity) {
      $data = new ResourceObjectData([]);
      return $this->createJsonapiResponse($data, $request, JsonResponse::HTTP_NOT_FOUND);
    }

    $data = $this->createIndividualDataFromEntity($this->entity);
    $response = $this->createJsonapiResponse($data, $request);

    $cacheable_metadata = new CacheableMetadata();
    $cacheable_metadata->addCacheContexts(['url.query_args:current_page']);
    $response->addCacheableDependency($cacheable_metadata);

    return $response;
  }

  /**
   * Gets the resource types for a route.
   *
   * Resource types are defined on routes in the `_jsonapi_resource_types`
   * route's defaults.
   *
   * @param \Symfony\Component\Routing\Route $route
   *   The route.
   * @param string $route_name
   *   The route name.
   *
   * @return array
   *   The resource types for the route.
   */
  public function getRouteResourceTypes(Route $route, string $route_name): array {
    $jsonapi_resource_types = [];
    if ($this->entity) {
      $entity_type_id = $this->entity->getEntityTypeId();
      $bundle = $this->entity->bundle();
      $jsonapi_resource_types = ["$entity_type_id--$bundle"];
    }

    return array_map(function ($resource_type_name) use ($route_name) {
      $resource_type = $this->resourceTypeRepository->getByTypeName($resource_type_name);
      if (is_null($resource_type)) {
        throw new RouteDefinitionException("The $route_name route definition's _jsonapi_resource_types route default declares the resource type $resource_type_name but a resource type by that name does not exist.");
      }
      return $resource_type;
    }, $jsonapi_resource_types ?: []);
  }

}
