<?php

namespace Drupal\Tests\deferjs\Unit\Form;

use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\deferjs\Form\DeferJsForm;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Tests class for \Drupal\deferjs\Form\DeferJsForm.
 *
 * @group deferjs
 * @group Fitch
 *
 * @coversDefaultClass \Drupal\deferjs\Form\DeferJsForm
 */
class DeferJsFormTest extends UnitTestCase {

  /**
   * The form under test.
   *
   * @var \Drupal\deferjs\Form\DeferJsForm
   */
  protected $form;

  /**
   * Mock entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * Mock config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;

  /**
   * Mock config object.
   *
   * @var \Drupal\Core\Config\Config|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $config;

  /**
   * Mock string translation.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $stringTranslation;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->config = $this->createMock(Config::class);
    $this->stringTranslation = $this->createMock(TranslationInterface::class);

    $this->configFactory->method('getEditable')
      ->with('deferjs.settings')
      ->willReturn($this->config);

    $this->stringTranslation->method('translate')
      ->willReturnCallback(function ($string, $args = [], $options = []) {
        return $string;
      });

    $this->form = new DeferJsForm($this->entityTypeManager);
    $this->form->setConfigFactory($this->configFactory);
    $this->form->setStringTranslation($this->stringTranslation);
  }

  /**
   * Tests the create method.
   *
   * @covers ::create
   */
  public function testCreate(): void {
    $container = $this->createMock(ContainerInterface::class);
    $container->expects($this->once())
      ->method('get')
      ->with('entity_type.manager')
      ->willReturn($this->entityTypeManager);

    $form = DeferJsForm::create($container);
    $this->assertInstanceOf(DeferJsForm::class, $form);
  }

  /**
   * Tests getFormId method.
   *
   * @covers ::getFormId
   */
  public function testGetFormId(): void {
    $this->assertEquals('deferjs_form', $this->form->getFormId());
  }

  /**
   * Tests the constructor.
   *
   * @covers ::__construct
   */
  public function testConstruct(): void {
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $form = new DeferJsForm($entityTypeManager);

    $reflection = new \ReflectionClass($form);
    $property = $reflection->getProperty('entityTypeManager');
    $property->setAccessible(TRUE);

    $this->assertSame($entityTypeManager, $property->getValue($form));
  }



  /**
   * Tests buildForm method.
   *
   * @covers ::buildForm
   */
  public function testBuildForm(): void {
    $form_state = $this->createMock(FormStateInterface::class);

    // Mock node types
    $nodeType1 = $this->getMockBuilder('\stdClass')
      ->addMethods(['id', 'label'])
      ->getMock();
    $nodeType1->method('id')->willReturn('article');
    $nodeType1->method('label')->willReturn('Article');

    $nodeType2 = $this->getMockBuilder('\stdClass')
      ->addMethods(['id', 'label'])
      ->getMock();
    $nodeType2->method('id')->willReturn('page');
    $nodeType2->method('label')->willReturn('Page');

    $nodeTypes = ['article' => $nodeType1, 'page' => $nodeType2];

    $storage = $this->createMock(EntityStorageInterface::class);
    $storage->method('loadMultiple')->willReturn($nodeTypes);

    $this->entityTypeManager->method('getStorage')
      ->with('node_type')
      ->willReturn($storage);

    // Mock config values
    $this->config->method('get')
      ->willReturnMap([
        ['module_enable', TRUE],
        ['enabled_content_types', ['article', 'page']],
        ['exclude_page', '/admin/*'],
        ['exclude_file', 'jquery.min.js'],
      ]);

    $form = $this->form->buildForm([], $form_state);

    // Assert form structure
    $this->assertArrayHasKey('settings', $form);
    $this->assertArrayHasKey('module_enable', $form['settings']);
    $this->assertArrayHasKey('enabled_content_types', $form['settings']);
    $this->assertArrayHasKey('exclude_page', $form['settings']);
    $this->assertArrayHasKey('exclude_file', $form['settings']);

    // Assert form element types
    $this->assertEquals('checkbox', $form['settings']['module_enable']['#type']);
    $this->assertEquals('checkboxes', $form['settings']['enabled_content_types']['#type']);
    $this->assertEquals('textarea', $form['settings']['exclude_page']['#type']);
    $this->assertEquals('textarea', $form['settings']['exclude_file']['#type']);

    // Assert default values
    $this->assertTrue($form['settings']['module_enable']['#default_value']);
    $this->assertEquals(['article', 'page'], $form['settings']['enabled_content_types']['#default_value']);
    $this->assertEquals('/admin/*', $form['settings']['exclude_page']['#default_value']);
    $this->assertEquals('jquery.min.js', $form['settings']['exclude_file']['#default_value']);

    // Assert content type options
    $options = $form['settings']['enabled_content_types']['#options'];
    $this->assertArrayHasKey('', $options);
    $this->assertArrayHasKey('article', $options);
    $this->assertArrayHasKey('page', $options);
    $this->assertEquals('Article', $options['article']);
    $this->assertEquals('Page', $options['page']);
  }

  /**
   * Tests submitForm method.
   *
   * @covers ::submitForm
   * @runInSeparateProcess
   */
  public function testSubmitForm(): void {
    if (!function_exists('Drupal\deferjs\Form\drupal_flush_all_caches')) {
      eval('namespace Drupal\deferjs\Form; function drupal_flush_all_caches() {}');
    }

    $messenger = $this->createMock(MessengerInterface::class);
    $container = $this->createMock(ContainerInterface::class);
    $container->method('get')
      ->with('messenger')
      ->willReturn($messenger);
    \Drupal::setContainer($container);

    $form_state = $this->createMock(FormStateInterface::class);

    $form_state->method('getValue')
      ->willReturnMap([
        ['module_enable', TRUE],
        ['enabled_content_types', ['article']],
        ['exclude_file', 'test.js'],
        ['exclude_page', '/test'],
      ]);

    $this->config->method('set')
      ->willReturnSelf();

    $this->config->expects($this->once())
      ->method('save')
      ->willReturnSelf();

    $form = [];
    $this->form->submitForm($form, $form_state);
  }

}
